"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UserActionFinder = void 0;
var _esQuery = require("@kbn/es-query");
var _api = require("../../../routes/api");
var _utils = require("../../../common/utils");
var _api2 = require("../../../../common/api");
var _constants = require("../../../../common/constants");
var _transform = require("../transform");
var _utils2 = require("../../../client/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class UserActionFinder {
  constructor(context) {
    this.context = context;
  }
  async find({
    caseId,
    sortOrder,
    types,
    page,
    perPage,
    filter
  }) {
    try {
      this.context.log.debug(`Attempting to find user actions for case id: ${caseId}`);
      const finalFilter = (0, _utils2.combineFilters)([filter, UserActionFinder.buildFilter(types)]);
      const userActions = await this.context.unsecuredSavedObjectsClient.find({
        type: _constants.CASE_USER_ACTION_SAVED_OBJECT,
        hasReference: {
          type: _constants.CASE_SAVED_OBJECT,
          id: caseId
        },
        page: page !== null && page !== void 0 ? page : _api.DEFAULT_PAGE,
        perPage: perPage !== null && perPage !== void 0 ? perPage : _api.DEFAULT_PER_PAGE,
        sortField: 'created_at',
        sortOrder: sortOrder !== null && sortOrder !== void 0 ? sortOrder : 'asc',
        filter: finalFilter
      });
      return (0, _transform.transformFindResponseToExternalModel)(userActions, this.context.persistableStateAttachmentTypeRegistry);
    } catch (error) {
      this.context.log.error(`Error finding user actions for case id: ${caseId}: ${error}`);
      throw error;
    }
  }
  static buildFilter(types = []) {
    const filters = types.map(type => UserActionFinder.buildFilterType(type));
    return (0, _utils2.combineFilters)(filters, _utils2.NodeBuilderOperators.or);
  }
  static buildFilterType(type) {
    switch (type) {
      case 'action':
        return UserActionFinder.buildActionFilter();
      case 'user':
        return UserActionFinder.buildCommentTypeFilter();
      case 'alert':
        return UserActionFinder.buildAlertCommentTypeFilter();
      case 'attachment':
        return UserActionFinder.buildAttachmentsFilter();
      default:
        return UserActionFinder.buildGenericTypeFilter(type);
    }
  }
  static buildActionFilter() {
    const filterForUserActionsExcludingComment = (0, _esQuery.fromKueryExpression)(`not ${_constants.CASE_USER_ACTION_SAVED_OBJECT}.attributes.payload.comment.type: ${_api2.CommentType.user}`);
    return filterForUserActionsExcludingComment;
  }
  static buildCommentTypeFilter() {
    return (0, _utils2.combineFilters)([(0, _utils2.buildFilter)({
      filters: [_api2.ActionTypes.comment],
      field: 'type',
      operator: 'or',
      type: _constants.CASE_USER_ACTION_SAVED_OBJECT
    }), (0, _utils2.buildFilter)({
      filters: [_api2.CommentType.user],
      field: 'payload.comment.type',
      operator: 'or',
      type: _constants.CASE_USER_ACTION_SAVED_OBJECT
    })], _utils2.NodeBuilderOperators.and);
  }
  static buildAlertCommentTypeFilter() {
    return (0, _utils2.combineFilters)([(0, _utils2.buildFilter)({
      filters: [_api2.ActionTypes.comment],
      field: 'type',
      operator: 'or',
      type: _constants.CASE_USER_ACTION_SAVED_OBJECT
    }), (0, _utils2.buildFilter)({
      filters: [_api2.CommentType.alert],
      field: 'payload.comment.type',
      operator: 'or',
      type: _constants.CASE_USER_ACTION_SAVED_OBJECT
    })], _utils2.NodeBuilderOperators.and);
  }
  static buildAttachmentsFilter() {
    return (0, _utils2.combineFilters)([(0, _utils2.buildFilter)({
      filters: [_api2.ActionTypes.comment],
      field: 'type',
      operator: 'or',
      type: _constants.CASE_USER_ACTION_SAVED_OBJECT
    }), (0, _utils2.buildFilter)({
      filters: [_api2.CommentType.persistableState, _api2.CommentType.externalReference],
      field: 'payload.comment.type',
      operator: 'or',
      type: _constants.CASE_USER_ACTION_SAVED_OBJECT
    })], _utils2.NodeBuilderOperators.and);
  }
  static buildGenericTypeFilter(type) {
    return (0, _utils2.buildFilter)({
      filters: [type],
      field: 'type',
      operator: 'or',
      type: _constants.CASE_USER_ACTION_SAVED_OBJECT
    });
  }
  async findStatusChanges({
    caseId,
    filter
  }) {
    try {
      this.context.log.debug('Attempting to find status changes');
      const updateActionFilter = (0, _utils2.buildFilter)({
        filters: _api2.Actions.update,
        field: 'action',
        operator: 'or',
        type: _constants.CASE_USER_ACTION_SAVED_OBJECT
      });
      const statusChangeFilter = (0, _utils2.buildFilter)({
        filters: _api2.ActionTypes.status,
        field: 'type',
        operator: 'or',
        type: _constants.CASE_USER_ACTION_SAVED_OBJECT
      });
      const combinedFilters = (0, _utils2.combineFilters)([updateActionFilter, statusChangeFilter, filter]);
      const finder = this.context.unsecuredSavedObjectsClient.createPointInTimeFinder({
        type: _constants.CASE_USER_ACTION_SAVED_OBJECT,
        hasReference: {
          type: _constants.CASE_SAVED_OBJECT,
          id: caseId
        },
        sortField: _utils.defaultSortField,
        sortOrder: 'asc',
        filter: combinedFilters,
        perPage: _constants.MAX_DOCS_PER_PAGE
      });
      let userActions = [];
      for await (const findResults of finder.find()) {
        userActions = userActions.concat(findResults.saved_objects.map(so => (0, _transform.transformToExternalModel)(so, this.context.persistableStateAttachmentTypeRegistry)));
      }
      return userActions;
    } catch (error) {
      this.context.log.error(`Error finding status changes: ${error}`);
      throw error;
    }
  }
}
exports.UserActionFinder = UserActionFinder;