"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkGet = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _pipeable = require("fp-ts/lib/pipeable");
var _Either = require("fp-ts/lib/Either");
var _function = require("fp-ts/lib/function");
var _lodash = require("lodash");
var _constants = require("../../../common/constants");
var _api = require("../../../common/api");
var _runtime_types = require("../../../common/api/runtime_types");
var _error = require("../../common/error");
var _utils = require("../../common/utils");
var _utils2 = require("../../authorization/utils");
var _authorization = require("../../authorization");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Retrieves multiple cases by ids.
 */
const bulkGet = async (params, clientArgs) => {
  const {
    services: {
      caseService,
      attachmentService
    },
    logger,
    authorization
  } = clientArgs;
  try {
    const fields = (0, _utils2.includeFieldsRequiredForAuthentication)((0, _utils.asArray)(params.fields));
    const request = (0, _pipeable.pipe)((0, _api.excess)(_api.CasesBulkGetRequestRt).decode({
      ...params,
      fields
    }), (0, _Either.fold)((0, _api.throwErrors)(_boom.default.badRequest), _function.identity));
    throwErrorIfCaseIdsReachTheLimit(request.ids);
    throwErrorIfFieldsAreInvalid(fields);
    const cases = await caseService.getCases({
      caseIds: request.ids,
      fields
    });
    const [validCases, soBulkGetErrors] = (0, _lodash.partition)(cases.saved_objects, caseInfo => caseInfo.error === undefined);
    const {
      authorized: authorizedCases,
      unauthorized: unauthorizedCases
    } = await authorization.getAndEnsureAuthorizedEntities({
      savedObjects: validCases,
      operation: _authorization.Operations.bulkGetCases
    });
    const requestForTotals = ['totalComment', 'totalAlerts'].some(totalKey => !fields || fields.includes(totalKey));
    const commentTotals = requestForTotals ? await attachmentService.getter.getCaseCommentStats({
      caseIds: authorizedCases.map(theCase => theCase.id)
    }) : new Map();
    const flattenedCases = authorizedCases.map(theCase => {
      var _commentTotals$get;
      const {
        alerts,
        userComments
      } = (_commentTotals$get = commentTotals.get(theCase.id)) !== null && _commentTotals$get !== void 0 ? _commentTotals$get : {
        alerts: 0,
        userComments: 0
      };
      const flattenedCase = (0, _utils.flattenCaseSavedObject)({
        savedObject: theCase,
        totalComment: userComments,
        totalAlerts: alerts
      });
      if (!(fields !== null && fields !== void 0 && fields.length)) {
        return flattenedCase;
      }
      return (0, _lodash.pick)(flattenedCase, [...fields, 'id', 'version']);
    });
    const typeToEncode = (0, _api.getTypeForCertainFieldsFromArray)(_api.CasesResponseRt, fields);
    const casesToReturn = typeToEncode.encode(flattenedCases);
    const errors = constructErrors(soBulkGetErrors, unauthorizedCases);
    return {
      cases: casesToReturn,
      errors
    };
  } catch (error) {
    var _params$ids;
    const ids = (_params$ids = params.ids) !== null && _params$ids !== void 0 ? _params$ids : [];
    throw (0, _error.createCaseError)({
      message: `Failed to bulk get cases: ${ids.join(', ')}: ${error}`,
      error,
      logger
    });
  }
};
exports.bulkGet = bulkGet;
const throwErrorIfFieldsAreInvalid = fields => {
  var _getTypeProps;
  if (!fields || fields.length === 0) {
    return;
  }
  const typeProps = (_getTypeProps = (0, _runtime_types.getTypeProps)(_api.CaseResponseRt)) !== null && _getTypeProps !== void 0 ? _getTypeProps : {};
  const validFields = Object.keys(typeProps);
  for (const field of fields) {
    if (!validFields.includes(field)) {
      throw _boom.default.badRequest(`Field: ${field} is not supported`);
    }
  }
};
const throwErrorIfCaseIdsReachTheLimit = ids => {
  if (ids.length > _constants.MAX_BULK_GET_CASES) {
    throw _boom.default.badRequest(`Maximum request limit of ${_constants.MAX_BULK_GET_CASES} cases reached`);
  }
};
const constructErrors = (soBulkGetErrors, unauthorizedCases) => {
  const errors = [];
  for (const soError of soBulkGetErrors) {
    errors.push({
      error: soError.error.error,
      message: soError.error.message,
      status: soError.error.statusCode,
      caseId: soError.id
    });
  }
  for (const theCase of unauthorizedCases) {
    errors.push({
      error: 'Forbidden',
      message: `Unauthorized to access case with owner: "${theCase.attributes.owner}"`,
      status: 403,
      caseId: theCase.id
    });
  }
  return errors;
};