"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.assetsRoutes = assetsRoutes;
var _configSchema = require("@kbn/config-schema");
var _lodash = require("lodash");
var _debug_log = require("../../common/debug_log");
var _constants = require("../constants");
var _get_assets = require("../lib/get_assets");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const assetType = _configSchema.schema.oneOf([_configSchema.schema.literal('k8s.pod'), _configSchema.schema.literal('k8s.cluster'), _configSchema.schema.literal('k8s.node')]);
const getAssetsQueryOptions = _configSchema.schema.object({
  from: _configSchema.schema.maybe(_configSchema.schema.string()),
  to: _configSchema.schema.maybe(_configSchema.schema.string()),
  type: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(assetType), assetType])),
  ean: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()])),
  size: _configSchema.schema.maybe(_configSchema.schema.number())
});
function assetsRoutes({
  router
}) {
  // GET /assets
  router.get({
    path: `${_constants.ASSET_MANAGER_API_BASE}/assets`,
    validate: {
      query: getAssetsQueryOptions
    }
  }, async (context, req, res) => {
    const {
      size,
      ...filters
    } = req.query || {};
    if (filters.type && filters.ean) {
      return res.badRequest({
        body: 'Filters "type" and "ean" are mutually exclusive but found both.'
      });
    }
    const esClient = await (0, _utils.getEsClientFromContext)(context);
    try {
      const results = await (0, _get_assets.getAssets)({
        esClient,
        size,
        filters
      });
      return res.ok({
        body: {
          results
        }
      });
    } catch (error) {
      (0, _debug_log.debug)('error looking up asset records', error);
      return res.customError({
        statusCode: 500
      });
    }
  });

  // GET /assets/diff
  const getAssetsDiffQueryOptions = _configSchema.schema.object({
    aFrom: _configSchema.schema.string(),
    aTo: _configSchema.schema.string(),
    bFrom: _configSchema.schema.string(),
    bTo: _configSchema.schema.string(),
    type: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(assetType), assetType]))
  });
  router.get({
    path: `${_constants.ASSET_MANAGER_API_BASE}/assets/diff`,
    validate: {
      query: getAssetsDiffQueryOptions
    }
  }, async (context, req, res) => {
    const {
      aFrom,
      aTo,
      bFrom,
      bTo,
      type
    } = req.query;
    if (new Date(aFrom) > new Date(aTo)) {
      return res.badRequest({
        body: `Time range cannot move backwards in time. "aTo" (${aTo}) is before "aFrom" (${aFrom}).`
      });
    }
    if (new Date(bFrom) > new Date(bTo)) {
      return res.badRequest({
        body: `Time range cannot move backwards in time. "bTo" (${bTo}) is before "bFrom" (${bFrom}).`
      });
    }
    const esClient = await (0, _utils.getEsClientFromContext)(context);
    try {
      const resultsForA = await (0, _get_assets.getAssets)({
        esClient,
        filters: {
          from: aFrom,
          to: aTo,
          type
        }
      });
      const resultsForB = await (0, _get_assets.getAssets)({
        esClient,
        filters: {
          from: bFrom,
          to: bTo,
          type
        }
      });
      const onlyInA = (0, _lodash.differenceBy)(resultsForA, resultsForB, 'asset.ean');
      const onlyInB = (0, _lodash.differenceBy)(resultsForB, resultsForA, 'asset.ean');
      const inBoth = (0, _lodash.intersectionBy)(resultsForA, resultsForB, 'asset.ean');
      return res.ok({
        body: {
          onlyInA,
          onlyInB,
          inBoth
        }
      });
    } catch (error) {
      (0, _debug_log.debug)('error looking up asset records', error);
      return res.customError({
        statusCode: 500
      });
    }
  });
}