"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getLatencyPeriods = getLatencyPeriods;
exports.getLatencyTimeseries = getLatencyTimeseries;
var _server = require("@kbn/observability-plugin/server");
var _document_type = require("../../../../common/document_type");
var _apm = require("../../../../common/es_fields/apm");
var _latency_aggregation_types = require("../../../../common/latency_aggregation_types");
var _environment_query = require("../../../../common/utils/environment_query");
var _get_offset_in_ms = require("../../../../common/utils/get_offset_in_ms");
var _offset_previous_period_coordinate = require("../../../../common/utils/offset_previous_period_coordinate");
var _latency_aggregation_type = require("../../../lib/helpers/latency_aggregation_type");
var _transactions = require("../../../lib/helpers/transactions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function searchLatency({
  environment,
  kuery,
  serviceName,
  transactionType,
  transactionName,
  apmEventClient,
  latencyAggregationType,
  start,
  end,
  offset,
  serverlessId,
  documentType,
  rollupInterval,
  bucketSizeInSeconds
}) {
  const {
    startWithOffset,
    endWithOffset
  } = (0, _get_offset_in_ms.getOffsetInMs)({
    start,
    end,
    offset
  });
  const transactionDurationField = documentType === _document_type.ApmDocumentType.ServiceTransactionMetric && latencyAggregationType !== _latency_aggregation_types.LatencyAggregationType.avg ? _apm.TRANSACTION_DURATION_HISTOGRAM : (0, _transactions.getDurationFieldForTransactions)(documentType);
  const params = {
    apm: {
      sources: [{
        documentType,
        rollupInterval
      }]
    },
    body: {
      track_total_hits: false,
      size: 0,
      query: {
        bool: {
          filter: [{
            term: {
              [_apm.SERVICE_NAME]: serviceName
            }
          }, ...(0, _server.rangeQuery)(startWithOffset, endWithOffset), ...(0, _environment_query.environmentQuery)(environment), ...(0, _server.kqlQuery)(kuery), ...(0, _server.termQuery)(_apm.TRANSACTION_NAME, transactionName), ...(0, _server.termQuery)(_apm.TRANSACTION_TYPE, transactionType), ...(0, _server.termQuery)(_apm.FAAS_ID, serverlessId)]
        }
      },
      aggs: {
        latencyTimeseries: {
          date_histogram: {
            field: '@timestamp',
            fixed_interval: `${bucketSizeInSeconds}s`,
            min_doc_count: 0,
            extended_bounds: {
              min: startWithOffset,
              max: endWithOffset
            }
          },
          aggs: (0, _latency_aggregation_type.getLatencyAggregation)(latencyAggregationType, transactionDurationField)
        },
        overall_avg_duration: {
          avg: {
            field: transactionDurationField
          }
        }
      }
    }
  };
  return apmEventClient.search('get_latency_charts', params);
}
async function getLatencyTimeseries({
  environment,
  kuery,
  serviceName,
  transactionType,
  transactionName,
  apmEventClient,
  latencyAggregationType,
  start,
  end,
  offset,
  serverlessId,
  documentType,
  rollupInterval,
  bucketSizeInSeconds
}) {
  const response = await searchLatency({
    environment,
    kuery,
    serviceName,
    transactionType,
    transactionName,
    apmEventClient,
    latencyAggregationType,
    start,
    end,
    offset,
    serverlessId,
    documentType,
    rollupInterval,
    bucketSizeInSeconds
  });
  if (!response.aggregations) {
    return {
      latencyTimeseries: [],
      overallAvgDuration: null
    };
  }
  return {
    overallAvgDuration: response.aggregations.overall_avg_duration.value || null,
    latencyTimeseries: response.aggregations.latencyTimeseries.buckets.map(bucket => {
      return {
        x: bucket.key,
        y: (0, _latency_aggregation_type.getLatencyValue)({
          latencyAggregationType,
          aggregation: bucket.latency
        })
      };
    })
  };
}
async function getLatencyPeriods({
  serviceName,
  transactionType,
  transactionName,
  apmEventClient,
  latencyAggregationType,
  kuery,
  environment,
  start,
  end,
  offset,
  documentType,
  rollupInterval,
  bucketSizeInSeconds
}) {
  const options = {
    serviceName,
    transactionType,
    transactionName,
    apmEventClient,
    kuery,
    environment,
    documentType,
    rollupInterval,
    bucketSizeInSeconds
  };
  const currentPeriodPromise = getLatencyTimeseries({
    ...options,
    start,
    end,
    latencyAggregationType: latencyAggregationType
  });
  const previousPeriodPromise = offset ? getLatencyTimeseries({
    ...options,
    start,
    end,
    offset,
    latencyAggregationType: latencyAggregationType
  }) : {
    latencyTimeseries: [],
    overallAvgDuration: null
  };
  const [currentPeriod, previousPeriod] = await Promise.all([currentPeriodPromise, previousPeriodPromise]);
  return {
    currentPeriod,
    previousPeriod: {
      ...previousPeriod,
      latencyTimeseries: (0, _offset_previous_period_coordinate.offsetPreviousPeriodCoordinates)({
        currentPeriodTimeseries: currentPeriod.latencyTimeseries,
        previousPeriodTimeseries: previousPeriod.latencyTimeseries
      })
    }
  };
}