"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.tasks = void 0;
var _esQuery = require("@kbn/es-query");
var _common = require("@kbn/observability-plugin/common");
var _crypto = require("crypto");
var _lodash = require("lodash");
var _agent_name = require("../../../../common/agent_name");
var _apm = require("../../../../common/es_fields/apm");
var _service_groups = require("../../../../common/service_groups");
var _as_mutable_array = require("../../../../common/utils/as_mutable_array");
var _get_kuery_fields = require("../../../../common/utils/get_kuery_fields");
var _get_apm_indices = require("../../../routes/settings/apm_indices/get_apm_indices");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TIME_RANGES = ['1d', 'all'];
const range1d = {
  range: {
    '@timestamp': {
      gte: 'now-1d'
    }
  }
};
const timeout = '5m';
const tasks = [{
  name: 'aggregated_transactions',
  // Record the number of metric documents we can expect in different scenarios. We simulate this by requesting data for 1m,
  // adding a composite aggregation on a number of fields and counting the number of buckets. The resulting count is an
  // approximation of the amount of metric documents that will be created. We record both the expected metric document count plus
  // the transaction count for that time range.
  executor: async ({
    indices,
    telemetryClient
  }) => {
    async function getBucketCountFromPaginatedQuery(sources, prevResult, after) {
      // eslint-disable-next-line @typescript-eslint/naming-convention
      let {
        expected_metric_document_count
      } = prevResult !== null && prevResult !== void 0 ? prevResult : {
        transaction_count: 0,
        expected_metric_document_count: 0
      };
      const params = {
        index: [indices.transaction],
        body: {
          track_total_hits: true,
          size: 0,
          timeout,
          query: {
            bool: {
              filter: [{
                term: {
                  [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.transaction
                }
              }, {
                range: {
                  '@timestamp': {
                    gte: start,
                    lt: end
                  }
                }
              }]
            }
          },
          aggs: {
            transaction_metric_groups: {
              composite: {
                ...(after ? {
                  after
                } : {}),
                size: 10000,
                sources: sources.map((source, index) => {
                  return {
                    [index]: source
                  };
                })
              }
            }
          }
        }
      };
      const result = await telemetryClient.search(params);
      let nextAfter;
      if (result.aggregations) {
        nextAfter = result.aggregations.transaction_metric_groups.after_key;
        expected_metric_document_count += result.aggregations.transaction_metric_groups.buckets.length;
      }
      const transactionCount = result.hits.total.value;
      if (nextAfter) {
        return await getBucketCountFromPaginatedQuery(sources, {
          expected_metric_document_count,
          transaction_count: transactionCount
        }, nextAfter);
      }
      return {
        expected_metric_document_count,
        transaction_count: transactionCount,
        ratio: expected_metric_document_count / transactionCount
      };
    }

    // fixed date range for reliable results
    const lastTransaction = (await telemetryClient.search({
      index: indices.transaction,
      body: {
        timeout,
        query: {
          bool: {
            filter: [{
              term: {
                [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.transaction
              }
            }]
          }
        },
        size: 1,
        track_total_hits: false,
        sort: {
          '@timestamp': 'desc'
        }
      }
    })).hits.hits[0];
    if (!lastTransaction) {
      return {};
    }
    const end = new Date(lastTransaction._source['@timestamp']).getTime() - 5 * 60 * 1000;
    const start = end - 60 * 1000;
    const simpleTermFields = [_apm.TRANSACTION_NAME, _apm.TRANSACTION_RESULT, _apm.TRANSACTION_TYPE, _apm.AGENT_NAME, _apm.SERVICE_ENVIRONMENT, _apm.SERVICE_VERSION, _apm.HOST_NAME, _apm.CONTAINER_ID, _apm.KUBERNETES_POD_NAME].map(field => ({
      terms: {
        field,
        missing_bucket: true
      }
    }));
    const observerHostname = {
      terms: {
        field: _apm.OBSERVER_HOSTNAME,
        missing_bucket: true
      }
    };
    const baseFields = [...simpleTermFields,
    // user_agent.name only for page-load transactions
    {
      terms: {
        script: `
              if (doc['transaction.type'].value == 'page-load' && doc['user_agent.name'].size() > 0) {
                return doc['user_agent.name'].value;
              }

              return null;
            `,
        missing_bucket: true
      }
    },
    // transaction.root
    {
      terms: {
        script: `return doc['parent.id'].size() == 0`,
        missing_bucket: true
      }
    }];
    const results = {
      current_implementation: await getBucketCountFromPaginatedQuery([...baseFields, observerHostname]),
      with_country: await getBucketCountFromPaginatedQuery([...baseFields, observerHostname, {
        terms: {
          script: `
                if (doc['transaction.type'].value == 'page-load' && doc['client.geo.country_iso_code'].size() > 0) {
                  return doc['client.geo.country_iso_code'].value;
                }
                return null;
              `,
          missing_bucket: true
        }
      }]),
      no_observer_name: await getBucketCountFromPaginatedQuery(baseFields)
    };
    return {
      aggregated_transactions: results
    };
  }
}, {
  name: 'cloud',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    function getBucketKeys({
      buckets
    }) {
      return buckets.map(bucket => bucket.key);
    }
    const az = 'availability_zone';
    const region = 'region';
    const provider = 'provider';
    const response = await telemetryClient.search({
      index: [indices.error, indices.metric, indices.span, indices.transaction],
      body: {
        track_total_hits: false,
        size: 0,
        timeout,
        aggs: {
          [az]: {
            terms: {
              field: _apm.CLOUD_AVAILABILITY_ZONE
            }
          },
          [provider]: {
            terms: {
              field: _apm.CLOUD_PROVIDER
            }
          },
          [region]: {
            terms: {
              field: _apm.CLOUD_REGION
            }
          }
        }
      }
    });
    const {
      aggregations
    } = response;
    if (!aggregations) {
      return {
        cloud: {
          [az]: [],
          [provider]: [],
          [region]: []
        }
      };
    }
    const cloud = {
      [az]: getBucketKeys(aggregations[az]),
      [provider]: getBucketKeys(aggregations[provider]),
      [region]: getBucketKeys(aggregations[region])
    };
    return {
      cloud
    };
  }
}, {
  name: 'host',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    function getBucketKeys({
      buckets
    }) {
      return buckets.map(bucket => bucket.key);
    }
    const response = await telemetryClient.search({
      index: [indices.error, indices.metric, indices.span, indices.transaction],
      body: {
        track_total_hits: false,
        size: 0,
        timeout,
        aggs: {
          platform: {
            terms: {
              field: _apm.HOST_OS_PLATFORM
            }
          }
        }
      }
    });
    const {
      aggregations
    } = response;
    if (!aggregations) {
      return {
        host: {
          os: {
            platform: []
          }
        }
      };
    }
    const host = {
      os: {
        platform: getBucketKeys(aggregations.platform)
      }
    };
    return {
      host
    };
  }
}, {
  name: 'environments',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4;
    const response = await telemetryClient.search({
      index: [indices.transaction],
      body: {
        track_total_hits: false,
        size: 0,
        timeout,
        query: {
          bool: {
            filter: [range1d]
          }
        },
        aggs: {
          environments: {
            terms: {
              field: _apm.SERVICE_ENVIRONMENT,
              size: 5
            }
          },
          service_environments: {
            composite: {
              size: 1000,
              sources: (0, _as_mutable_array.asMutableArray)([{
                [_apm.SERVICE_ENVIRONMENT]: {
                  terms: {
                    field: _apm.SERVICE_ENVIRONMENT,
                    missing_bucket: true
                  }
                }
              }, {
                [_apm.SERVICE_NAME]: {
                  terms: {
                    field: _apm.SERVICE_NAME
                  }
                }
              }])
            }
          }
        }
      }
    });
    const topEnvironments = (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.environments.buckets.map(bucket => bucket.key)) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
    const serviceEnvironments = {};
    const buckets = (_response$aggregation3 = (_response$aggregation4 = response.aggregations) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.service_environments.buckets) !== null && _response$aggregation3 !== void 0 ? _response$aggregation3 : [];
    buckets.forEach(bucket => {
      var _serviceEnvironments$;
      const serviceName = bucket.key['service.name'];
      const environment = bucket.key['service.environment'];
      const environments = (_serviceEnvironments$ = serviceEnvironments[serviceName]) !== null && _serviceEnvironments$ !== void 0 ? _serviceEnvironments$ : [];
      serviceEnvironments[serviceName] = environments.concat(environment);
    });
    const servicesWithoutEnvironment = Object.keys((0, _lodash.pickBy)(serviceEnvironments, environments => environments.includes(null)));
    const servicesWithMultipleEnvironments = Object.keys((0, _lodash.pickBy)(serviceEnvironments, environments => environments.length > 1));
    return {
      environments: {
        services_without_environment: servicesWithoutEnvironment.length,
        services_with_multiple_environments: servicesWithMultipleEnvironments.length,
        top_environments: topEnvironments
      }
    };
  }
}, {
  name: 'processor_events',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    const indicesByProcessorEvent = {
      error: indices.error,
      metric: indices.metric,
      span: indices.span,
      transaction: indices.transaction,
      onboarding: indices.onboarding
    };
    const events = Object.keys(indicesByProcessorEvent);
    const jobs = events.flatMap(processorEvent => TIME_RANGES.map(timeRange => ({
      processorEvent,
      timeRange
    })));
    const allData = await jobs.reduce((prevJob, current) => {
      return prevJob.then(async data => {
        var _retainmentResponse$h;
        const {
          processorEvent,
          timeRange
        } = current;
        const totalHitsResponse = await telemetryClient.search({
          index: indicesByProcessorEvent[processorEvent],
          body: {
            size: 0,
            track_total_hits: true,
            timeout,
            query: {
              bool: {
                filter: [{
                  term: {
                    [_apm.PROCESSOR_EVENT]: processorEvent
                  }
                }, ...(timeRange === '1d' ? [range1d] : [])]
              }
            }
          }
        });
        const retainmentResponse = timeRange === 'all' ? await telemetryClient.search({
          index: indicesByProcessorEvent[processorEvent],
          size: 10,
          body: {
            track_total_hits: false,
            size: 0,
            timeout,
            query: {
              bool: {
                filter: [{
                  term: {
                    [_apm.PROCESSOR_EVENT]: processorEvent
                  }
                }]
              }
            },
            sort: {
              '@timestamp': 'asc'
            },
            _source: ['@timestamp']
          }
        }) : null;
        const event = retainmentResponse === null || retainmentResponse === void 0 ? void 0 : (_retainmentResponse$h = retainmentResponse.hits.hits[0]) === null || _retainmentResponse$h === void 0 ? void 0 : _retainmentResponse$h._source;
        return (0, _lodash.merge)({}, data, {
          counts: {
            [processorEvent]: {
              [timeRange]: totalHitsResponse.hits.total.value
            }
          },
          ...(event ? {
            retainment: {
              [processorEvent]: {
                ms: new Date().getTime() - new Date(event['@timestamp']).getTime()
              }
            }
          } : {})
        });
      });
    }, Promise.resolve({}));
    return allData;
  }
}, {
  name: 'agent_configuration',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    const agentConfigurationCount = await telemetryClient.search({
      index: _get_apm_indices.APM_AGENT_CONFIGURATION_INDEX,
      body: {
        size: 0,
        timeout,
        track_total_hits: true
      }
    });
    return {
      counts: {
        agent_configuration: {
          all: agentConfigurationCount.hits.total.value
        }
      }
    };
  }
}, {
  name: 'services',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    const servicesPerAgent = await _agent_name.AGENT_NAMES.reduce((prevJob, agentName) => {
      return prevJob.then(async data => {
        var _response$aggregation5;
        const response = await telemetryClient.search({
          index: [indices.error, indices.span, indices.metric, indices.transaction],
          body: {
            size: 0,
            track_total_hits: false,
            timeout,
            query: {
              bool: {
                filter: [{
                  term: {
                    [_apm.AGENT_NAME]: agentName
                  }
                }, range1d]
              }
            },
            aggs: {
              services: {
                cardinality: {
                  field: _apm.SERVICE_NAME
                }
              }
            }
          }
        });
        return {
          ...data,
          [agentName]: ((_response$aggregation5 = response.aggregations) === null || _response$aggregation5 === void 0 ? void 0 : _response$aggregation5.services.value) || 0
        };
      });
    }, Promise.resolve({}));
    return {
      has_any_services: (0, _lodash.sum)(Object.values(servicesPerAgent)) > 0,
      services_per_agent: servicesPerAgent
    };
  }
}, {
  name: 'versions',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _response$hits$hits$, _hit$observer;
    const response = await telemetryClient.search({
      index: [indices.transaction, indices.span, indices.error],
      terminate_after: 1,
      body: {
        query: {
          exists: {
            field: 'observer.version'
          }
        },
        track_total_hits: false,
        size: 1,
        timeout,
        sort: {
          '@timestamp': 'desc'
        }
      }
    });
    const hit = (_response$hits$hits$ = response.hits.hits[0]) === null || _response$hits$hits$ === void 0 ? void 0 : _response$hits$hits$._source;
    if (!hit || !((_hit$observer = hit.observer) !== null && _hit$observer !== void 0 && _hit$observer.version)) {
      return {};
    }
    const [major, minor, patch] = hit.observer.version.split('.').map(part => Number(part));
    return {
      version: {
        apm_server: {
          major,
          minor,
          patch
        }
      }
    };
  }
}, {
  name: 'groupings',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _await$telemetryClien, _await$telemetryClien2, _await$telemetryClien3, _await$telemetryClien4, _await$telemetryClien5;
    const errorGroupsCount = (_await$telemetryClien = (await telemetryClient.search({
      index: indices.error,
      body: {
        size: 0,
        timeout,
        track_total_hits: false,
        query: {
          bool: {
            filter: [{
              term: {
                [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.error
              }
            }, range1d]
          }
        },
        aggs: {
          top_service: {
            terms: {
              field: _apm.SERVICE_NAME,
              order: {
                error_groups: 'desc'
              },
              size: 1
            },
            aggs: {
              error_groups: {
                cardinality: {
                  field: _apm.ERROR_GROUP_ID
                }
              }
            }
          }
        }
      }
    })).aggregations) === null || _await$telemetryClien === void 0 ? void 0 : (_await$telemetryClien2 = _await$telemetryClien.top_service.buckets[0]) === null || _await$telemetryClien2 === void 0 ? void 0 : _await$telemetryClien2.error_groups.value;
    const transactionGroupsCount = (_await$telemetryClien3 = (await telemetryClient.search({
      index: indices.transaction,
      body: {
        track_total_hits: false,
        size: 0,
        timeout,
        query: {
          bool: {
            filter: [{
              term: {
                [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.transaction
              }
            }, range1d]
          }
        },
        aggs: {
          top_service: {
            terms: {
              field: _apm.SERVICE_NAME,
              order: {
                transaction_groups: 'desc'
              },
              size: 1
            },
            aggs: {
              transaction_groups: {
                cardinality: {
                  field: _apm.TRANSACTION_NAME
                }
              }
            }
          }
        }
      }
    })).aggregations) === null || _await$telemetryClien3 === void 0 ? void 0 : (_await$telemetryClien4 = _await$telemetryClien3.top_service.buckets[0]) === null || _await$telemetryClien4 === void 0 ? void 0 : _await$telemetryClien4.transaction_groups.value;
    const tracesPerDayCount = (await telemetryClient.search({
      index: indices.transaction,
      body: {
        query: {
          bool: {
            filter: [{
              term: {
                [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.transaction
              }
            }, range1d],
            must_not: {
              exists: {
                field: _apm.PARENT_ID
              }
            }
          }
        },
        track_total_hits: true,
        size: 0,
        timeout
      }
    })).hits.total.value;
    const servicesCount = (_await$telemetryClien5 = (await telemetryClient.search({
      index: [indices.transaction, indices.error, indices.metric],
      body: {
        track_total_hits: false,
        size: 0,
        timeout,
        query: {
          bool: {
            filter: [range1d]
          }
        },
        aggs: {
          service_name: {
            cardinality: {
              field: _apm.SERVICE_NAME
            }
          }
        }
      }
    })).aggregations) === null || _await$telemetryClien5 === void 0 ? void 0 : _await$telemetryClien5.service_name.value;
    return {
      counts: {
        max_error_groups_per_service: {
          '1d': errorGroupsCount || 0
        },
        max_transaction_groups_per_service: {
          '1d': transactionGroupsCount || 0
        },
        traces: {
          '1d': tracesPerDayCount || 0
        },
        services: {
          '1d': servicesCount || 0
        }
      }
    };
  }
}, {
  name: 'integrations',
  executor: async ({
    telemetryClient
  }) => {
    var _response$body$count, _response$body;
    const apmJobs = ['apm-*', '*-high_mean_response_time'];
    const response = await telemetryClient.transportRequest({
      method: 'get',
      path: `/_ml/anomaly_detectors/${apmJobs.join(',')}`
    });
    return {
      integrations: {
        ml: {
          all_jobs_count: (_response$body$count = (_response$body = response.body) === null || _response$body === void 0 ? void 0 : _response$body.count) !== null && _response$body$count !== void 0 ? _response$body$count : 0
        }
      }
    };
  }
}, {
  name: 'agents',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    const size = 3;
    const agentData = await _agent_name.AGENT_NAMES.reduce(async (prevJob, agentName) => {
      const data = await prevJob;
      const response = await telemetryClient.search({
        index: [indices.error, indices.metric, indices.transaction],
        body: {
          track_total_hits: false,
          size: 0,
          timeout,
          query: {
            bool: {
              filter: [{
                term: {
                  [_apm.AGENT_NAME]: agentName
                }
              }, range1d]
            }
          },
          sort: {
            '@timestamp': 'desc'
          },
          aggs: {
            [_apm.AGENT_ACTIVATION_METHOD]: {
              terms: {
                field: _apm.AGENT_ACTIVATION_METHOD,
                size
              }
            },
            [_apm.AGENT_VERSION]: {
              terms: {
                field: _apm.AGENT_VERSION,
                size
              }
            },
            [_apm.SERVICE_FRAMEWORK_NAME]: {
              terms: {
                field: _apm.SERVICE_FRAMEWORK_NAME,
                size
              },
              aggs: {
                [_apm.SERVICE_FRAMEWORK_VERSION]: {
                  terms: {
                    field: _apm.SERVICE_FRAMEWORK_VERSION,
                    size
                  }
                }
              }
            },
            [_apm.SERVICE_FRAMEWORK_VERSION]: {
              terms: {
                field: _apm.SERVICE_FRAMEWORK_VERSION,
                size
              }
            },
            [_apm.SERVICE_LANGUAGE_NAME]: {
              terms: {
                field: _apm.SERVICE_LANGUAGE_NAME,
                size
              },
              aggs: {
                [_apm.SERVICE_LANGUAGE_VERSION]: {
                  terms: {
                    field: _apm.SERVICE_LANGUAGE_VERSION,
                    size
                  }
                }
              }
            },
            [_apm.SERVICE_LANGUAGE_VERSION]: {
              terms: {
                field: _apm.SERVICE_LANGUAGE_VERSION,
                size
              }
            },
            [_apm.SERVICE_RUNTIME_NAME]: {
              terms: {
                field: _apm.SERVICE_RUNTIME_NAME,
                size
              },
              aggs: {
                [_apm.SERVICE_RUNTIME_VERSION]: {
                  terms: {
                    field: _apm.SERVICE_RUNTIME_VERSION,
                    size
                  }
                }
              }
            },
            [_apm.SERVICE_RUNTIME_VERSION]: {
              terms: {
                field: _apm.SERVICE_RUNTIME_VERSION,
                size
              }
            }
          }
        }
      });
      const {
        aggregations
      } = response;
      if (!aggregations) {
        return data;
      }
      const toComposite = (outerKey, innerKey) => `${outerKey}/${innerKey}`;
      return {
        ...data,
        [agentName]: {
          agent: {
            activation_method: aggregations[_apm.AGENT_ACTIVATION_METHOD].buckets.map(bucket => bucket.key).slice(0, size),
            version: aggregations[_apm.AGENT_VERSION].buckets.map(bucket => bucket.key)
          },
          service: {
            framework: {
              name: aggregations[_apm.SERVICE_FRAMEWORK_NAME].buckets.map(bucket => bucket.key).slice(0, size),
              version: aggregations[_apm.SERVICE_FRAMEWORK_VERSION].buckets.map(bucket => bucket.key).slice(0, size),
              composite: (0, _lodash.sortBy)((0, _lodash.flatten)(aggregations[_apm.SERVICE_FRAMEWORK_NAME].buckets.map(bucket => bucket[_apm.SERVICE_FRAMEWORK_VERSION].buckets.map(versionBucket => ({
                doc_count: versionBucket.doc_count,
                name: toComposite(bucket.key, versionBucket.key)
              })))), 'doc_count').reverse().slice(0, size).map(composite => composite.name)
            },
            language: {
              name: aggregations[_apm.SERVICE_LANGUAGE_NAME].buckets.map(bucket => bucket.key).slice(0, size),
              version: aggregations[_apm.SERVICE_LANGUAGE_VERSION].buckets.map(bucket => bucket.key).slice(0, size),
              composite: (0, _lodash.sortBy)((0, _lodash.flatten)(aggregations[_apm.SERVICE_LANGUAGE_NAME].buckets.map(bucket => bucket[_apm.SERVICE_LANGUAGE_VERSION].buckets.map(versionBucket => ({
                doc_count: versionBucket.doc_count,
                name: toComposite(bucket.key, versionBucket.key)
              })))), 'doc_count').reverse().slice(0, size).map(composite => composite.name)
            },
            runtime: {
              name: aggregations[_apm.SERVICE_RUNTIME_NAME].buckets.map(bucket => bucket.key).slice(0, size),
              version: aggregations[_apm.SERVICE_RUNTIME_VERSION].buckets.map(bucket => bucket.key).slice(0, size),
              composite: (0, _lodash.sortBy)((0, _lodash.flatten)(aggregations[_apm.SERVICE_RUNTIME_NAME].buckets.map(bucket => bucket[_apm.SERVICE_RUNTIME_VERSION].buckets.map(versionBucket => ({
                doc_count: versionBucket.doc_count,
                name: toComposite(bucket.key, versionBucket.key)
              })))), 'doc_count').reverse().slice(0, size).map(composite => composite.name)
            }
          }
        }
      };
    }, Promise.resolve({}));
    return {
      agents: agentData
    };
  }
}, {
  name: 'indices_stats',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _metricIndicesRespons, _metricIndicesRespons2, _metricIndicesRespons3, _metricIndicesRespons4, _metricIndicesRespons5, _metricIndicesRespons6, _metricIndicesRespons7, _metricIndicesRespons8, _metricIndicesRespons9, _metricIndicesRespons10, _tracesIndicesRespons, _tracesIndicesRespons2, _tracesIndicesRespons3, _tracesIndicesRespons4, _tracesIndicesRespons5, _tracesIndicesRespons6, _tracesIndicesRespons7, _tracesIndicesRespons8, _tracesIndicesRespons9, _tracesIndicesRespons10, _response$_shards$tot, _response$_shards, _response$_all$total$, _response$_all, _response$_all$total, _response$_all$total$2, _response$_all$total$3, _response$_all2, _response$_all2$total, _response$_all2$total2;
    const response = await telemetryClient.indicesStats({
      index: [_get_apm_indices.APM_AGENT_CONFIGURATION_INDEX, indices.error, indices.metric, indices.onboarding, indices.span, indices.transaction]
    });
    const metricIndicesResponse = await telemetryClient.indicesStats({
      index: [indices.metric]
    });
    const tracesIndicesResponse = await telemetryClient.indicesStats({
      index: [indices.span, indices.transaction]
    });
    return {
      indices: {
        metric: {
          shards: {
            total: (_metricIndicesRespons = (_metricIndicesRespons2 = metricIndicesResponse._shards) === null || _metricIndicesRespons2 === void 0 ? void 0 : _metricIndicesRespons2.total) !== null && _metricIndicesRespons !== void 0 ? _metricIndicesRespons : 0
          },
          all: {
            total: {
              docs: {
                count: (_metricIndicesRespons3 = (_metricIndicesRespons4 = metricIndicesResponse._all) === null || _metricIndicesRespons4 === void 0 ? void 0 : (_metricIndicesRespons5 = _metricIndicesRespons4.total) === null || _metricIndicesRespons5 === void 0 ? void 0 : (_metricIndicesRespons6 = _metricIndicesRespons5.docs) === null || _metricIndicesRespons6 === void 0 ? void 0 : _metricIndicesRespons6.count) !== null && _metricIndicesRespons3 !== void 0 ? _metricIndicesRespons3 : 0
              },
              store: {
                size_in_bytes: (_metricIndicesRespons7 = (_metricIndicesRespons8 = metricIndicesResponse._all) === null || _metricIndicesRespons8 === void 0 ? void 0 : (_metricIndicesRespons9 = _metricIndicesRespons8.total) === null || _metricIndicesRespons9 === void 0 ? void 0 : (_metricIndicesRespons10 = _metricIndicesRespons9.store) === null || _metricIndicesRespons10 === void 0 ? void 0 : _metricIndicesRespons10.size_in_bytes) !== null && _metricIndicesRespons7 !== void 0 ? _metricIndicesRespons7 : 0
              }
            }
          }
        },
        traces: {
          shards: {
            total: (_tracesIndicesRespons = (_tracesIndicesRespons2 = tracesIndicesResponse._shards) === null || _tracesIndicesRespons2 === void 0 ? void 0 : _tracesIndicesRespons2.total) !== null && _tracesIndicesRespons !== void 0 ? _tracesIndicesRespons : 0
          },
          all: {
            total: {
              docs: {
                count: (_tracesIndicesRespons3 = (_tracesIndicesRespons4 = tracesIndicesResponse._all) === null || _tracesIndicesRespons4 === void 0 ? void 0 : (_tracesIndicesRespons5 = _tracesIndicesRespons4.total) === null || _tracesIndicesRespons5 === void 0 ? void 0 : (_tracesIndicesRespons6 = _tracesIndicesRespons5.docs) === null || _tracesIndicesRespons6 === void 0 ? void 0 : _tracesIndicesRespons6.count) !== null && _tracesIndicesRespons3 !== void 0 ? _tracesIndicesRespons3 : 0
              },
              store: {
                size_in_bytes: (_tracesIndicesRespons7 = (_tracesIndicesRespons8 = tracesIndicesResponse._all) === null || _tracesIndicesRespons8 === void 0 ? void 0 : (_tracesIndicesRespons9 = _tracesIndicesRespons8.total) === null || _tracesIndicesRespons9 === void 0 ? void 0 : (_tracesIndicesRespons10 = _tracesIndicesRespons9.store) === null || _tracesIndicesRespons10 === void 0 ? void 0 : _tracesIndicesRespons10.size_in_bytes) !== null && _tracesIndicesRespons7 !== void 0 ? _tracesIndicesRespons7 : 0
              }
            }
          }
        },
        shards: {
          total: (_response$_shards$tot = (_response$_shards = response._shards) === null || _response$_shards === void 0 ? void 0 : _response$_shards.total) !== null && _response$_shards$tot !== void 0 ? _response$_shards$tot : 0
        },
        all: {
          total: {
            docs: {
              count: (_response$_all$total$ = (_response$_all = response._all) === null || _response$_all === void 0 ? void 0 : (_response$_all$total = _response$_all.total) === null || _response$_all$total === void 0 ? void 0 : (_response$_all$total$2 = _response$_all$total.docs) === null || _response$_all$total$2 === void 0 ? void 0 : _response$_all$total$2.count) !== null && _response$_all$total$ !== void 0 ? _response$_all$total$ : 0
            },
            store: {
              size_in_bytes: (_response$_all$total$3 = (_response$_all2 = response._all) === null || _response$_all2 === void 0 ? void 0 : (_response$_all2$total = _response$_all2.total) === null || _response$_all2$total === void 0 ? void 0 : (_response$_all2$total2 = _response$_all2$total.store) === null || _response$_all2$total2 === void 0 ? void 0 : _response$_all2$total2.size_in_bytes) !== null && _response$_all$total$3 !== void 0 ? _response$_all$total$3 : 0
            }
          }
        }
      }
    };
  }
}, {
  name: 'cardinality',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _rumAgentCardinalityR, _allAgentsCardinality, _rumAgentCardinalityR2, _allAgentsCardinality2, _rumAgentCardinalityR3;
    const allAgentsCardinalityResponse = await telemetryClient.search({
      index: [indices.transaction],
      body: {
        track_total_hits: false,
        size: 0,
        timeout,
        query: {
          bool: {
            filter: [range1d]
          }
        },
        aggs: {
          [_apm.TRANSACTION_NAME]: {
            cardinality: {
              field: _apm.TRANSACTION_NAME
            }
          },
          [_apm.USER_AGENT_ORIGINAL]: {
            cardinality: {
              field: _apm.USER_AGENT_ORIGINAL
            }
          }
        }
      }
    });
    const rumAgentCardinalityResponse = await telemetryClient.search({
      index: [indices.transaction],
      body: {
        track_total_hits: false,
        size: 0,
        timeout,
        query: {
          bool: {
            filter: [range1d, {
              terms: {
                [_apm.AGENT_NAME]: _agent_name.RUM_AGENT_NAMES
              }
            }]
          }
        },
        aggs: {
          [_apm.CLIENT_GEO_COUNTRY_ISO_CODE]: {
            cardinality: {
              field: _apm.CLIENT_GEO_COUNTRY_ISO_CODE
            }
          },
          [_apm.TRANSACTION_NAME]: {
            cardinality: {
              field: _apm.TRANSACTION_NAME
            }
          },
          [_apm.USER_AGENT_ORIGINAL]: {
            cardinality: {
              field: _apm.USER_AGENT_ORIGINAL
            }
          }
        }
      }
    });
    return {
      cardinality: {
        client: {
          geo: {
            country_iso_code: {
              rum: {
                '1d': (_rumAgentCardinalityR = rumAgentCardinalityResponse.aggregations) === null || _rumAgentCardinalityR === void 0 ? void 0 : _rumAgentCardinalityR[_apm.CLIENT_GEO_COUNTRY_ISO_CODE].value
              }
            }
          }
        },
        transaction: {
          name: {
            all_agents: {
              '1d': (_allAgentsCardinality = allAgentsCardinalityResponse.aggregations) === null || _allAgentsCardinality === void 0 ? void 0 : _allAgentsCardinality[_apm.TRANSACTION_NAME].value
            },
            rum: {
              '1d': (_rumAgentCardinalityR2 = rumAgentCardinalityResponse.aggregations) === null || _rumAgentCardinalityR2 === void 0 ? void 0 : _rumAgentCardinalityR2[_apm.TRANSACTION_NAME].value
            }
          }
        },
        user_agent: {
          original: {
            all_agents: {
              '1d': (_allAgentsCardinality2 = allAgentsCardinalityResponse.aggregations) === null || _allAgentsCardinality2 === void 0 ? void 0 : _allAgentsCardinality2[_apm.USER_AGENT_ORIGINAL].value
            },
            rum: {
              '1d': (_rumAgentCardinalityR3 = rumAgentCardinalityResponse.aggregations) === null || _rumAgentCardinalityR3 === void 0 ? void 0 : _rumAgentCardinalityR3[_apm.USER_AGENT_ORIGINAL].value
            }
          }
        }
      }
    };
  }
}, {
  name: 'service_groups',
  executor: async ({
    savedObjectsClient
  }) => {
    var _response$total;
    const response = await savedObjectsClient.find({
      type: _service_groups.APM_SERVICE_GROUP_SAVED_OBJECT_TYPE,
      page: 1,
      perPage: _service_groups.MAX_NUMBER_OF_SERVICE_GROUPS,
      sortField: 'updated_at',
      sortOrder: 'desc',
      namespaces: ['*']
    });
    const kueryNodes = response.saved_objects.map(({
      attributes: {
        kuery
      }
    }) => (0, _esQuery.fromKueryExpression)(kuery));
    const kueryFields = (0, _get_kuery_fields.getKueryFields)(kueryNodes);
    return {
      service_groups: {
        kuery_fields: (0, _lodash.uniq)(kueryFields),
        total: (_response$total = response.total) !== null && _response$total !== void 0 ? _response$total : 0
      }
    };
  }
}, {
  name: 'per_service',
  executor: async ({
    indices,
    telemetryClient
  }) => {
    var _response$aggregation6, _response$aggregation7;
    const response = await telemetryClient.search({
      index: [indices.transaction],
      body: {
        track_total_hits: false,
        size: 0,
        timeout,
        query: {
          bool: {
            filter: [{
              range: {
                '@timestamp': {
                  gte: 'now-1h'
                }
              }
            }, {
              term: {
                [_apm.PROCESSOR_EVENT]: _common.ProcessorEvent.transaction
              }
            }]
          }
        },
        aggs: {
          service_names: {
            terms: {
              field: _apm.SERVICE_NAME,
              size: 2500
            },
            aggs: {
              environments: {
                terms: {
                  field: _apm.SERVICE_ENVIRONMENT,
                  size: 5
                },
                aggs: {
                  instances: {
                    cardinality: {
                      field: _apm.SERVICE_NODE_NAME
                    }
                  },
                  transaction_types: {
                    cardinality: {
                      field: _apm.TRANSACTION_TYPE
                    }
                  },
                  top_metrics: {
                    top_metrics: {
                      sort: '_score',
                      metrics: [{
                        field: _apm.AGENT_ACTIVATION_METHOD
                      }, {
                        field: _apm.AGENT_NAME
                      }, {
                        field: _apm.AGENT_VERSION
                      }, {
                        field: _apm.SERVICE_LANGUAGE_NAME
                      }, {
                        field: _apm.SERVICE_LANGUAGE_VERSION
                      }, {
                        field: _apm.SERVICE_FRAMEWORK_NAME
                      }, {
                        field: _apm.SERVICE_FRAMEWORK_VERSION
                      }, {
                        field: _apm.SERVICE_RUNTIME_NAME
                      }, {
                        field: _apm.SERVICE_RUNTIME_VERSION
                      }, {
                        field: _apm.KUBERNETES_POD_NAME
                      }, {
                        field: _apm.CONTAINER_ID
                      }]
                    }
                  },
                  [_apm.CLOUD_REGION]: {
                    terms: {
                      field: _apm.CLOUD_REGION,
                      size: 5
                    }
                  },
                  [_apm.CLOUD_PROVIDER]: {
                    terms: {
                      field: _apm.CLOUD_PROVIDER,
                      size: 3
                    }
                  },
                  [_apm.CLOUD_AVAILABILITY_ZONE]: {
                    terms: {
                      field: _apm.CLOUD_AVAILABILITY_ZONE,
                      size: 5
                    }
                  },
                  [_apm.FAAS_TRIGGER_TYPE]: {
                    terms: {
                      field: _apm.FAAS_TRIGGER_TYPE,
                      size: 5
                    }
                  }
                }
              }
            }
          }
        }
      }
    });
    const serviceBuckets = (_response$aggregation6 = (_response$aggregation7 = response.aggregations) === null || _response$aggregation7 === void 0 ? void 0 : _response$aggregation7.service_names.buckets) !== null && _response$aggregation6 !== void 0 ? _response$aggregation6 : [];
    const data = serviceBuckets.flatMap(serviceBucket => {
      var _serviceBucket$enviro, _serviceBucket$enviro2;
      const envHash = (0, _crypto.createHash)('sha256').update(serviceBucket.key).digest('hex');
      const envBuckets = (_serviceBucket$enviro = (_serviceBucket$enviro2 = serviceBucket.environments) === null || _serviceBucket$enviro2 === void 0 ? void 0 : _serviceBucket$enviro2.buckets) !== null && _serviceBucket$enviro !== void 0 ? _serviceBucket$enviro : [];
      return envBuckets.map(envBucket => {
        var _envBucket$instances$, _envBucket$transactio, _envBucket$CLOUD_AVAI, _envBucket$CLOUD_AVAI2, _envBucket$CLOUD_REGI, _envBucket$CLOUD_REGI2, _envBucket$CLOUD_PROV, _envBucket$CLOUD_PROV2, _envBucket$FAAS_TRIGG, _envBucket$FAAS_TRIGG2, _envBucket$top_metric, _envBucket$top_metric2, _envBucket$top_metric3, _envBucket$top_metric4, _envBucket$top_metric5, _envBucket$top_metric6, _envBucket$top_metric7, _envBucket$top_metric8, _envBucket$top_metric9, _envBucket$top_metric10, _envBucket$top_metric11;
        const nameHash = (0, _crypto.createHash)('sha256').update(envBucket.key).digest('hex');
        const fullServiceName = `${nameHash}~${envHash}`;
        return {
          service_id: fullServiceName,
          timed_out: response.timed_out,
          num_service_nodes: (_envBucket$instances$ = envBucket.instances.value) !== null && _envBucket$instances$ !== void 0 ? _envBucket$instances$ : 1,
          num_transaction_types: (_envBucket$transactio = envBucket.transaction_types.value) !== null && _envBucket$transactio !== void 0 ? _envBucket$transactio : 0,
          cloud: {
            availability_zones: (_envBucket$CLOUD_AVAI = (_envBucket$CLOUD_AVAI2 = envBucket[_apm.CLOUD_AVAILABILITY_ZONE]) === null || _envBucket$CLOUD_AVAI2 === void 0 ? void 0 : _envBucket$CLOUD_AVAI2.buckets.map(inner => inner.key)) !== null && _envBucket$CLOUD_AVAI !== void 0 ? _envBucket$CLOUD_AVAI : [],
            regions: (_envBucket$CLOUD_REGI = (_envBucket$CLOUD_REGI2 = envBucket[_apm.CLOUD_REGION]) === null || _envBucket$CLOUD_REGI2 === void 0 ? void 0 : _envBucket$CLOUD_REGI2.buckets.map(inner => inner.key)) !== null && _envBucket$CLOUD_REGI !== void 0 ? _envBucket$CLOUD_REGI : [],
            providers: (_envBucket$CLOUD_PROV = (_envBucket$CLOUD_PROV2 = envBucket[_apm.CLOUD_PROVIDER]) === null || _envBucket$CLOUD_PROV2 === void 0 ? void 0 : _envBucket$CLOUD_PROV2.buckets.map(inner => inner.key)) !== null && _envBucket$CLOUD_PROV !== void 0 ? _envBucket$CLOUD_PROV : []
          },
          faas: {
            trigger: {
              type: (_envBucket$FAAS_TRIGG = (_envBucket$FAAS_TRIGG2 = envBucket[_apm.FAAS_TRIGGER_TYPE]) === null || _envBucket$FAAS_TRIGG2 === void 0 ? void 0 : _envBucket$FAAS_TRIGG2.buckets.map(inner => inner.key)) !== null && _envBucket$FAAS_TRIGG !== void 0 ? _envBucket$FAAS_TRIGG : []
            }
          },
          agent: {
            name: (_envBucket$top_metric = envBucket.top_metrics) === null || _envBucket$top_metric === void 0 ? void 0 : _envBucket$top_metric.top[0].metrics[_apm.AGENT_NAME],
            activation_method: (_envBucket$top_metric2 = envBucket.top_metrics) === null || _envBucket$top_metric2 === void 0 ? void 0 : _envBucket$top_metric2.top[0].metrics[_apm.AGENT_ACTIVATION_METHOD],
            version: (_envBucket$top_metric3 = envBucket.top_metrics) === null || _envBucket$top_metric3 === void 0 ? void 0 : _envBucket$top_metric3.top[0].metrics[_apm.AGENT_VERSION]
          },
          service: {
            language: {
              name: (_envBucket$top_metric4 = envBucket.top_metrics) === null || _envBucket$top_metric4 === void 0 ? void 0 : _envBucket$top_metric4.top[0].metrics[_apm.SERVICE_LANGUAGE_NAME],
              version: (_envBucket$top_metric5 = envBucket.top_metrics) === null || _envBucket$top_metric5 === void 0 ? void 0 : _envBucket$top_metric5.top[0].metrics[_apm.SERVICE_LANGUAGE_VERSION]
            },
            framework: {
              name: (_envBucket$top_metric6 = envBucket.top_metrics) === null || _envBucket$top_metric6 === void 0 ? void 0 : _envBucket$top_metric6.top[0].metrics[_apm.SERVICE_FRAMEWORK_NAME],
              version: (_envBucket$top_metric7 = envBucket.top_metrics) === null || _envBucket$top_metric7 === void 0 ? void 0 : _envBucket$top_metric7.top[0].metrics[_apm.SERVICE_FRAMEWORK_VERSION]
            },
            runtime: {
              name: (_envBucket$top_metric8 = envBucket.top_metrics) === null || _envBucket$top_metric8 === void 0 ? void 0 : _envBucket$top_metric8.top[0].metrics[_apm.SERVICE_RUNTIME_NAME],
              version: (_envBucket$top_metric9 = envBucket.top_metrics) === null || _envBucket$top_metric9 === void 0 ? void 0 : _envBucket$top_metric9.top[0].metrics[_apm.SERVICE_RUNTIME_VERSION]
            }
          },
          kubernetes: {
            pod: {
              name: (_envBucket$top_metric10 = envBucket.top_metrics) === null || _envBucket$top_metric10 === void 0 ? void 0 : _envBucket$top_metric10.top[0].metrics[_apm.KUBERNETES_POD_NAME]
            }
          },
          container: {
            id: (_envBucket$top_metric11 = envBucket.top_metrics) === null || _envBucket$top_metric11 === void 0 ? void 0 : _envBucket$top_metric11.top[0].metrics[_apm.CONTAINER_ID]
          }
        };
      });
    });
    return {
      per_service: data
    };
  }
}];
exports.tasks = tasks;