"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchIndexNameAPI = fetchIndexNameAPI;
exports.fetchTriggeredAlertsHistory = fetchTriggeredAlertsHistory;
exports.useFetchTriggeredAlertsHistory = useFetchTriggeredAlertsHistory;
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _common = require("@kbn/rule-registry-plugin/common");
var _react = require("react");
var _public = require("@kbn/kibana-react-plugin/public");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useFetchTriggeredAlertsHistory({
  features,
  ruleId
}) {
  const {
    http
  } = (0, _public.useKibana)().services;
  const [triggeredAlertsHistory, setTriggeredAlertsHistory] = (0, _react.useState)({
    isLoadingTriggeredAlertHistory: true
  });
  const isCancelledRef = (0, _react.useRef)(false);
  const abortCtrlRef = (0, _react.useRef)(new AbortController());
  const loadRuleAlertsAgg = (0, _react.useCallback)(async () => {
    isCancelledRef.current = false;
    abortCtrlRef.current.abort();
    abortCtrlRef.current = new AbortController();
    try {
      if (!http) throw new Error('No http client');
      if (!features) return;
      const {
        index
      } = await fetchIndexNameAPI({
        http,
        features
      });
      const {
        totalTriggeredAlerts,
        histogramTriggeredAlerts,
        error,
        avgTimeToRecoverUS
      } = await fetchTriggeredAlertsHistory({
        http,
        index,
        ruleId,
        signal: abortCtrlRef.current.signal
      });
      if (error) throw error;
      if (!isCancelledRef.current) {
        setTriggeredAlertsHistory(oldState => ({
          ...oldState,
          triggeredAlertsData: {
            totalTriggeredAlerts,
            histogramTriggeredAlerts,
            avgTimeToRecoverUS
          },
          isLoadingRuleAlertsAggs: false
        }));
      }
    } catch (error) {
      if (!isCancelledRef.current) {
        if (error.name !== 'AbortError') {
          setTriggeredAlertsHistory(oldState => ({
            ...oldState,
            isLoadingRuleAlertsAggs: false,
            errorTriggeredAlertHistory: error,
            triggeredAlertsData: undefined
          }));
        }
      }
    }
  }, [features, http, ruleId]);
  (0, _react.useEffect)(() => {
    loadRuleAlertsAgg();
  }, [loadRuleAlertsAgg]);
  return triggeredAlertsHistory;
}
async function fetchIndexNameAPI({
  http,
  features
}) {
  const res = await http.get(`${_common.BASE_RAC_ALERTS_API_PATH}/index`, {
    query: {
      features
    }
  });
  return {
    index: res.index_name[0]
  };
}
async function fetchTriggeredAlertsHistory({
  http,
  index,
  ruleId,
  signal
}) {
  try {
    var _res$aggregations, _res$aggregations2;
    const res = await http.post(`${_common.BASE_RAC_ALERTS_API_PATH}/find`, {
      signal,
      body: JSON.stringify({
        index,
        size: 0,
        query: {
          bool: {
            must: [{
              term: {
                [_ruleDataUtils.ALERT_RULE_UUID]: ruleId
              }
            }, {
              range: {
                [_ruleDataUtils.ALERT_TIME_RANGE]: {
                  gte: 'now-30d',
                  lt: 'now'
                }
              }
            }]
          }
        },
        aggs: {
          histogramTriggeredAlerts: {
            date_histogram: {
              field: _ruleDataUtils.ALERT_START,
              fixed_interval: '1d',
              extended_bounds: {
                min: 'now-30d',
                max: 'now'
              }
            }
          },
          avgTimeToRecoverUS: {
            filter: {
              term: {
                [_ruleDataUtils.ALERT_STATUS]: 'recovered'
              }
            },
            aggs: {
              recoveryTime: {
                avg: {
                  field: _ruleDataUtils.ALERT_DURATION
                }
              }
            }
          }
        }
      })
    });
    const totalTriggeredAlerts = res === null || res === void 0 ? void 0 : res.hits.total.value;
    const histogramTriggeredAlerts = res === null || res === void 0 ? void 0 : (_res$aggregations = res.aggregations) === null || _res$aggregations === void 0 ? void 0 : _res$aggregations.histogramTriggeredAlerts.buckets;
    const avgTimeToRecoverUS = res === null || res === void 0 ? void 0 : (_res$aggregations2 = res.aggregations) === null || _res$aggregations2 === void 0 ? void 0 : _res$aggregations2.avgTimeToRecoverUS.recoveryTime.value;
    return {
      totalTriggeredAlerts,
      histogramTriggeredAlerts,
      avgTimeToRecoverUS
    };
  } catch (error) {
    console.error(error);
    return {
      error,
      totalTriggeredAlerts: 0,
      histogramTriggeredAlerts: [],
      avgTimeToRecoverUS: 0
    };
  }
}