"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EQLCodeEditorCompleter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("./constants");
var _tokens = require("./tokens");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EQLCodeEditorCompleter {
  constructor() {
    (0, _defineProperty2.default)(this, "callback", void 0);
  }
  async getCompletionsAsync(session, position, prefix) {
    var _previousTokens$find$, _previousTokens$find;
    const token = session.getTokenAt(position.row, position.column);
    const tokensInLine = session.getTokens(position.row);
    function withWhitespace(vals, options = {}) {
      const {
        after = ' '
      } = options;
      let {
        before = ' '
      } = options;
      if (before && (token !== null && token !== void 0 && token.value.match(/^\s+$/) || token && token.type !== 'text')) {
        before = before.trimLeft();
      }
      return vals.map(val => {
        const suggestion = typeof val === 'string' ? {
          value: val
        } : val;
        const valueAsString = suggestion.value;
        return {
          ...suggestion,
          caption: valueAsString,
          value: [before, valueAsString, after].join('')
        };
      });
    }
    if (position.row === 0 && (!token || token.index === 0) && 'sequence by'.includes(prefix || '')) {
      return withWhitespace(['sequence by'], {
        before: '',
        after: ' '
      });
    }
    const previousTokens = tokensInLine.slice(0, token ? tokensInLine.indexOf(token) : tokensInLine.length).reverse();
    const completedEqlToken = previousTokens.find(t => t.type.startsWith('eql.'));
    switch (completedEqlToken === null || completedEqlToken === void 0 ? void 0 : completedEqlToken.type) {
      case undefined:
        return [...withWhitespace(['['], {
          before: '',
          after: ' '
        }), ...(position.row > 2 ? withWhitespace(['until'], {
          before: '',
          after: ' [ '
        }) : [])];
      case _tokens.EQLToken.Sequence:
        return withWhitespace(await this.getExternalSuggestions({
          type: _constants.EQLCodeEditorSuggestionType.Field
        }), {
          after: '\n\t[ '
        });
      case _tokens.EQLToken.SequenceItemStart:
        return withWhitespace([...(await this.getExternalSuggestions({
          type: _constants.EQLCodeEditorSuggestionType.EventType
        })), 'any'], {
          after: ' where '
        });
      case _tokens.EQLToken.EventType:
        return withWhitespace(['where']);
      case _tokens.EQLToken.Where:
      case _tokens.EQLToken.LogicalOperator:
        return [...withWhitespace(await this.getExternalSuggestions({
          type: _constants.EQLCodeEditorSuggestionType.Field
        })), ...withWhitespace(['true', 'false'], {
          after: ' ]\n\t'
        })];
      case _tokens.EQLToken.BoolCondition:
        return withWhitespace([']'], {
          after: '\n\t'
        });
      case _tokens.EQLToken.Operator:
      case _tokens.EQLToken.InOperator:
        const field = (_previousTokens$find$ = previousTokens === null || previousTokens === void 0 ? void 0 : (_previousTokens$find = previousTokens.find(t => t.type === _tokens.EQLToken.Field)) === null || _previousTokens$find === void 0 ? void 0 : _previousTokens$find.value) !== null && _previousTokens$find$ !== void 0 ? _previousTokens$find$ : '';
        const hasStartedValueLiteral = !!(prefix !== null && prefix !== void 0 && prefix.trim()) || (token === null || token === void 0 ? void 0 : token.value.trim()) === '"';
        return withWhitespace(await this.getExternalSuggestions({
          type: _constants.EQLCodeEditorSuggestionType.Value,
          field,
          value: prefix !== null && prefix !== void 0 ? prefix : ''
        }), {
          before: hasStartedValueLiteral ? '' : ' "',
          after: '" '
        });
      case _tokens.EQLToken.Value:
        return [...withWhitespace([']'], {
          after: '\n\t'
        }), ...withWhitespace(['and', 'or'])];
    }
    return [];
  }
  async getExternalSuggestions(request) {
    if (this.callback) {
      return this.callback(request);
    }
    return [];
  }
  getCompletions(_, session, position, prefix, cb) {
    this.getCompletionsAsync(session, position, prefix).then(suggestions => {
      cb(null, suggestions.map(sugg => {
        const suggestion = typeof sugg === 'string' ? {
          value: sugg,
          score: 1000
        } : {
          score: 1000,
          ...sugg
        };
        return suggestion;
      }));
    }).catch(cb);
  }
  setSuggestionCb(cb) {
    this.callback = cb;
  }
}
exports.EQLCodeEditorCompleter = EQLCodeEditorCompleter;