"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.disable = disable;
var _authorization = require("../../authorization");
var _retry_if_conflicts = require("../../lib/retry_if_conflicts");
var _audit_events = require("../common/audit_events");
var _lib = require("../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function disable(context, {
  id
}) {
  return await (0, _retry_if_conflicts.retryIfConflicts)(context.logger, `rulesClient.disable('${id}')`, async () => await disableWithOCC(context, {
    id
  }));
}
async function disableWithOCC(context, {
  id
}) {
  var _context$auditLogger2;
  let attributes;
  let version;
  let references;
  try {
    const decryptedAlert = await context.encryptedSavedObjectsClient.getDecryptedAsInternalUser('alert', id, {
      namespace: context.namespace
    });
    attributes = decryptedAlert.attributes;
    version = decryptedAlert.version;
    references = decryptedAlert.references;
  } catch (e) {
    context.logger.error(`disable(): Failed to load API key of alert ${id}: ${e.message}`);
    // Still attempt to load the attributes and version using SOC
    const alert = await context.unsecuredSavedObjectsClient.get('alert', id);
    attributes = alert.attributes;
    version = alert.version;
    references = alert.references;
  }
  await (0, _lib.recoverRuleAlerts)(context, id, attributes);
  try {
    await context.authorization.ensureAuthorized({
      ruleTypeId: attributes.alertTypeId,
      consumer: attributes.consumer,
      operation: _authorization.WriteOperations.Disable,
      entity: _authorization.AlertingAuthorizationEntity.Rule
    });
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.DISABLE,
      savedObject: {
        type: 'alert',
        id
      },
      error
    }));
    throw error;
  }
  (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.ruleAuditEvent)({
    action: _audit_events.RuleAuditAction.DISABLE,
    outcome: 'unknown',
    savedObject: {
      type: 'alert',
      id
    }
  }));
  context.ruleTypeRegistry.ensureRuleTypeEnabled(attributes.alertTypeId);
  if (attributes.enabled === true) {
    const migratedActions = await (0, _lib.migrateLegacyActions)(context, {
      ruleId: id,
      actions: attributes.actions,
      references,
      attributes
    });
    await context.unsecuredSavedObjectsClient.update('alert', id, (0, _lib.updateMeta)(context, {
      ...attributes,
      enabled: false,
      scheduledTaskId: attributes.scheduledTaskId === id ? attributes.scheduledTaskId : null,
      updatedBy: await context.getUserName(),
      updatedAt: new Date().toISOString(),
      nextRun: null,
      ...(migratedActions.hasLegacyActions ? {
        actions: migratedActions.resultedActions,
        throttle: undefined,
        notifyWhen: undefined
      } : {})
    }), {
      version,
      ...(migratedActions.hasLegacyActions ? {
        references: migratedActions.resultedReferences
      } : {})
    });

    // If the scheduledTaskId does not match the rule id, we should
    // remove the task, otherwise mark the task as disabled
    if (attributes.scheduledTaskId) {
      if (attributes.scheduledTaskId !== id) {
        await context.taskManager.removeIfExists(attributes.scheduledTaskId);
      } else {
        await context.taskManager.bulkDisable([attributes.scheduledTaskId]);
      }
    }
  }
}