"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkEdit = bulkEdit;
exports.bulkEditFieldsToExcludeFromRevisionUpdates = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _lodash = require("lodash");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _esQuery = require("@kbn/es-query");
var _types = require("../../types");
var _lib = require("../../lib");
var _authorization = require("../../authorization");
var _parse_duration = require("../../../common/parse_duration");
var _bulk_mark_api_keys_for_invalidation = require("../../invalidate_pending_api_keys/bulk_mark_api_keys_for_invalidation");
var _audit_events = require("../common/audit_events");
var _common = require("../common");
var _constants = require("../common/constants");
var _mapped_params_utils = require("../common/mapped_params_utils");
var _lib2 = require("../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bulkEditFieldsToExcludeFromRevisionUpdates = new Set(['snoozeSchedule', 'apiKey']);
exports.bulkEditFieldsToExcludeFromRevisionUpdates = bulkEditFieldsToExcludeFromRevisionUpdates;
async function bulkEdit(context, options) {
  const queryFilter = options.filter;
  const ids = options.ids;
  if (ids && queryFilter) {
    throw _boom.default.badRequest("Both 'filter' and 'ids' are supplied. Define either 'ids' or 'filter' properties in method arguments");
  }
  const qNodeQueryFilter = (0, _common.buildKueryNodeFilter)(queryFilter);
  const qNodeFilter = ids ? (0, _lib.convertRuleIdsToKueryNode)(ids) : qNodeQueryFilter;
  let authorizationTuple;
  try {
    authorizationTuple = await context.authorization.getFindAuthorizationFilter(_authorization.AlertingAuthorizationEntity.Rule, _constants.alertingAuthorizationFilterOpts);
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.BULK_EDIT,
      error
    }));
    throw error;
  }
  const {
    filter: authorizationFilter
  } = authorizationTuple;
  const qNodeFilterWithAuth = authorizationFilter && qNodeFilter ? _esQuery.nodeBuilder.and([qNodeFilter, authorizationFilter]) : qNodeFilter;
  const {
    aggregations,
    total
  } = await context.unsecuredSavedObjectsClient.find({
    filter: qNodeFilterWithAuth,
    page: 1,
    perPage: 0,
    type: 'alert',
    aggs: {
      alertTypeId: {
        multi_terms: {
          terms: [{
            field: 'alert.attributes.alertTypeId'
          }, {
            field: 'alert.attributes.consumer'
          }]
        }
      }
    }
  });
  if (total > _constants.MAX_RULES_NUMBER_FOR_BULK_OPERATION) {
    throw _boom.default.badRequest(`More than ${_constants.MAX_RULES_NUMBER_FOR_BULK_OPERATION} rules matched for bulk edit`);
  }
  const buckets = aggregations === null || aggregations === void 0 ? void 0 : aggregations.alertTypeId.buckets;
  if (buckets === undefined) {
    throw Error('No rules found for bulk edit');
  }
  await (0, _pMap.default)(buckets, async ({
    key: [ruleType, consumer]
  }) => {
    context.ruleTypeRegistry.ensureRuleTypeEnabled(ruleType);
    try {
      await context.authorization.ensureAuthorized({
        ruleTypeId: ruleType,
        consumer,
        operation: _authorization.WriteOperations.BulkEdit,
        entity: _authorization.AlertingAuthorizationEntity.Rule
      });
    } catch (error) {
      var _context$auditLogger2;
      (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.ruleAuditEvent)({
        action: _audit_events.RuleAuditAction.BULK_EDIT,
        error
      }));
      throw error;
    }
  }, {
    concurrency: _constants.RULE_TYPE_CHECKS_CONCURRENCY
  });
  const {
    apiKeysToInvalidate,
    results,
    errors,
    skipped
  } = await (0, _common.retryIfBulkEditConflicts)(context.logger, `rulesClient.update('operations=${JSON.stringify(options.operations)}, paramsModifier=${options.paramsModifier ? '[Function]' : undefined}', shouldIncrementRevision=${options.shouldIncrementRevision ? '[Function]' : undefined}')`, filterKueryNode => bulkEditOcc(context, {
    filter: filterKueryNode,
    operations: options.operations,
    paramsModifier: options.paramsModifier,
    shouldIncrementRevision: options.shouldIncrementRevision
  }), qNodeFilterWithAuth);
  if (apiKeysToInvalidate.length > 0) {
    await (0, _bulk_mark_api_keys_for_invalidation.bulkMarkApiKeysForInvalidation)({
      apiKeys: apiKeysToInvalidate
    }, context.logger, context.unsecuredSavedObjectsClient);
  }
  const updatedRules = results.map(({
    id,
    attributes,
    references
  }) => {
    return (0, _lib2.getAlertFromRaw)(context, id, attributes.alertTypeId, attributes, references, false, false, false, false);
  });
  await bulkUpdateSchedules(context, options.operations, updatedRules);
  return {
    rules: updatedRules,
    skipped,
    errors,
    total
  };
}
async function bulkEditOcc(context, {
  filter,
  operations,
  paramsModifier,
  shouldIncrementRevision
}) {
  const rulesFinder = await context.encryptedSavedObjectsClient.createPointInTimeFinderDecryptedAsInternalUser({
    filter,
    type: 'alert',
    perPage: 100,
    ...(context.namespace ? {
      namespaces: [context.namespace]
    } : undefined)
  });
  const rules = [];
  const skipped = [];
  const errors = [];
  const apiKeysMap = new Map();
  const username = await context.getUserName();
  for await (const response of rulesFinder.find()) {
    await (0, _pMap.default)(response.saved_objects, async rule => updateRuleAttributesAndParamsInMemory({
      context,
      rule,
      operations,
      paramsModifier,
      apiKeysMap,
      rules,
      skipped,
      errors,
      username,
      shouldIncrementRevision
    }), {
      concurrency: _constants.API_KEY_GENERATE_CONCURRENCY
    });
  }
  await rulesFinder.close();
  const {
    result,
    apiKeysToInvalidate
  } = rules.length > 0 ? await saveBulkUpdatedRules(context, rules, apiKeysMap) : {
    result: {
      saved_objects: []
    },
    apiKeysToInvalidate: []
  };
  return {
    apiKeysToInvalidate,
    resultSavedObjects: result.saved_objects,
    errors,
    rules,
    skipped
  };
}
async function bulkUpdateSchedules(context, operations, updatedRules) {
  const scheduleOperation = operations.find(operation => operation.field === 'schedule');
  if (!(scheduleOperation !== null && scheduleOperation !== void 0 && scheduleOperation.value)) {
    return;
  }
  const taskIds = updatedRules.reduce((acc, rule) => {
    if (rule.scheduledTaskId) {
      acc.push(rule.scheduledTaskId);
    }
    return acc;
  }, []);
  try {
    await context.taskManager.bulkUpdateSchedules(taskIds, scheduleOperation.value);
    context.logger.debug(`Successfully updated schedules for underlying tasks: ${taskIds.join(', ')}`);
  } catch (error) {
    context.logger.error(`Failure to update schedules for underlying tasks: ${taskIds.join(', ')}. TaskManager bulkUpdateSchedules failed with Error: ${error.message}`);
  }
}
async function updateRuleAttributesAndParamsInMemory({
  context,
  rule,
  operations,
  paramsModifier,
  apiKeysMap,
  rules,
  skipped,
  errors,
  username,
  shouldIncrementRevision = () => true
}) {
  try {
    if (rule.attributes.apiKey) {
      apiKeysMap.set(rule.id, {
        oldApiKey: rule.attributes.apiKey,
        oldApiKeyCreatedByUser: rule.attributes.apiKeyCreatedByUser
      });
    }
    const ruleType = context.ruleTypeRegistry.get(rule.attributes.alertTypeId);
    await ensureAuthorizationForBulkUpdate(context, operations, rule);

    // migrate legacy actions only for SIEM rules
    const migratedActions = await (0, _lib2.migrateLegacyActions)(context, {
      ruleId: rule.id,
      actions: rule.attributes.actions,
      references: rule.references,
      attributes: rule.attributes
    });
    if (migratedActions.hasLegacyActions) {
      rule.attributes.actions = migratedActions.resultedActions;
      rule.references = migratedActions.resultedReferences;
    }
    const {
      attributes,
      ruleActions,
      hasUpdateApiKeyOperation,
      isAttributesUpdateSkipped
    } = await getUpdatedAttributesFromOperations(context, operations, rule, ruleType);
    validateScheduleInterval(context, attributes.schedule.interval, ruleType.id, rule.id);
    const params = (0, _common.injectReferencesIntoParams)(rule.id, ruleType, attributes.params, rule.references || []);
    const {
      modifiedParams: ruleParams,
      isParamsUpdateSkipped
    } = paramsModifier ? await paramsModifier(params) : {
      modifiedParams: params,
      isParamsUpdateSkipped: true
    };

    // Increment revision if params ended up being modified AND it wasn't already incremented as part of attribute update
    if (shouldIncrementRevision(ruleParams) && !isParamsUpdateSkipped && rule.attributes.revision === attributes.revision) {
      attributes.revision += 1;
    }

    // If neither attributes nor parameters were updated, mark
    // the rule as skipped and continue to the next rule.
    if (isAttributesUpdateSkipped && isParamsUpdateSkipped) {
      skipped.push({
        id: rule.id,
        name: rule.attributes.name,
        skip_reason: 'RULE_NOT_MODIFIED'
      });
      return;
    }

    // validate rule params
    const validatedAlertTypeParams = (0, _lib.validateRuleTypeParams)(ruleParams, ruleType.validate.params);
    const validatedMutatedAlertTypeParams = (0, _lib.validateMutatedRuleTypeParams)(validatedAlertTypeParams, rule.attributes.params, ruleType.validate.params);
    const {
      actions: rawAlertActions,
      references,
      params: updatedParams
    } = await (0, _lib2.extractReferences)(context, ruleType, ruleActions.actions, validatedMutatedAlertTypeParams);
    const {
      apiKeyAttributes
    } = await prepareApiKeys(context, rule, ruleType, apiKeysMap, attributes, hasUpdateApiKeyOperation, username);
    const {
      updatedAttributes
    } = updateAttributes(context, attributes, apiKeyAttributes, updatedParams, rawAlertActions, username);
    rules.push({
      ...rule,
      references,
      attributes: updatedAttributes
    });
  } catch (error) {
    var _rule$attributes, _context$auditLogger3;
    errors.push({
      message: error.message,
      rule: {
        id: rule.id,
        name: (_rule$attributes = rule.attributes) === null || _rule$attributes === void 0 ? void 0 : _rule$attributes.name
      }
    });
    (_context$auditLogger3 = context.auditLogger) === null || _context$auditLogger3 === void 0 ? void 0 : _context$auditLogger3.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.BULK_EDIT,
      error
    }));
  }
}
async function ensureAuthorizationForBulkUpdate(context, operations, rule) {
  if (rule.attributes.actions.length === 0) {
    return;
  }
  for (const operation of operations) {
    const {
      field
    } = operation;
    if (field === 'snoozeSchedule' || field === 'apiKey') {
      try {
        await context.actionsAuthorization.ensureAuthorized('execute');
        break;
      } catch (error) {
        throw Error(`Rule not authorized for bulk ${field} update - ${error.message}`);
      }
    }
  }
}
async function getUpdatedAttributesFromOperations(context, operations, rule, ruleType) {
  let attributes = (0, _lodash.cloneDeep)(rule.attributes);
  let ruleActions = {
    actions: (0, _common.injectReferencesIntoActions)(rule.id, rule.attributes.actions || [], rule.references || [])
  };
  let hasUpdateApiKeyOperation = false;
  let isAttributesUpdateSkipped = true;
  for (const operation of operations) {
    // Check if the update should be skipped for the current action.
    // If it should, save the skip reasons in attributesUpdateSkipReasons
    // and continue to the next operation before without
    // the `isAttributesUpdateSkipped` flag to false.
    switch (operation.field) {
      case 'actions':
        {
          const updatedOperation = {
            ...operation,
            value: (0, _lib2.addGeneratedActionValues)(operation.value)
          };
          try {
            await (0, _lib2.validateActions)(context, ruleType, {
              ...attributes,
              actions: updatedOperation.value
            });
          } catch (e) {
            // If validateActions fails on the first attempt, it may be because of legacy rule-level frequency params
            attributes = await attemptToMigrateLegacyFrequency(context, updatedOperation, attributes, ruleType);
          }
          const {
            modifiedAttributes,
            isAttributeModified
          } = (0, _common.applyBulkEditOperation)(updatedOperation, ruleActions);
          if (isAttributeModified) {
            ruleActions = modifiedAttributes;
            isAttributesUpdateSkipped = false;
          }
          break;
        }
      case 'snoozeSchedule':
        {
          // Silently skip adding snooze or snooze schedules on security
          // rules until we implement snoozing of their rules
          if (rule.attributes.consumer === _ruleDataUtils.AlertConsumers.SIEM) {
            // While the rule is technically not updated, we are still marking
            // the rule as updated in case of snoozing, until support
            // for snoozing is added.
            isAttributesUpdateSkipped = false;
            break;
          }
          if (operation.operation === 'set') {
            const snoozeAttributes = (0, _common.getBulkSnoozeAttributes)(rule.attributes, operation.value);
            try {
              (0, _common.verifySnoozeScheduleLimit)(snoozeAttributes);
            } catch (error) {
              throw Error(`Error updating rule: could not add snooze - ${error.message}`);
            }
            attributes = {
              ...attributes,
              ...snoozeAttributes
            };
          }
          if (operation.operation === 'delete') {
            const idsToDelete = operation.value && [...operation.value];
            if ((idsToDelete === null || idsToDelete === void 0 ? void 0 : idsToDelete.length) === 0) {
              var _attributes$snoozeSch;
              (_attributes$snoozeSch = attributes.snoozeSchedule) === null || _attributes$snoozeSch === void 0 ? void 0 : _attributes$snoozeSch.forEach(schedule => {
                if (schedule.id) {
                  idsToDelete.push(schedule.id);
                }
              });
            }
            attributes = {
              ...attributes,
              ...(0, _common.getBulkUnsnoozeAttributes)(attributes, idsToDelete)
            };
          }
          isAttributesUpdateSkipped = false;
          break;
        }
      case 'apiKey':
        {
          hasUpdateApiKeyOperation = true;
          isAttributesUpdateSkipped = false;
          break;
        }
      default:
        {
          if (operation.field === 'schedule') {
            validateScheduleOperation(operation.value, attributes.actions, rule.id);
          }
          const {
            modifiedAttributes,
            isAttributeModified
          } = (0, _common.applyBulkEditOperation)(operation, rule.attributes);
          if (isAttributeModified) {
            attributes = {
              ...attributes,
              ...modifiedAttributes
            };
            isAttributesUpdateSkipped = false;
          }
        }
    }
    // Only increment revision if update wasn't skipped and `operation.field` should result in a revision increment
    if (!isAttributesUpdateSkipped && !bulkEditFieldsToExcludeFromRevisionUpdates.has(operation.field) && rule.attributes.revision - attributes.revision === 0) {
      attributes.revision += 1;
    }
  }
  return {
    attributes,
    ruleActions,
    hasUpdateApiKeyOperation,
    isAttributesUpdateSkipped
  };
}
function validateScheduleInterval(context, scheduleInterval, ruleTypeId, ruleId) {
  if (!scheduleInterval) {
    return;
  }
  const isIntervalInvalid = (0, _parse_duration.parseDuration)(scheduleInterval) < context.minimumScheduleIntervalInMs;
  if (isIntervalInvalid && context.minimumScheduleInterval.enforce) {
    throw Error(`Error updating rule: the interval is less than the allowed minimum interval of ${context.minimumScheduleInterval.value}`);
  } else if (isIntervalInvalid && !context.minimumScheduleInterval.enforce) {
    context.logger.warn(`Rule schedule interval (${scheduleInterval}) for "${ruleTypeId}" rule type with ID "${ruleId}" is less than the minimum value (${context.minimumScheduleInterval.value}). Running rules at this interval may impact alerting performance. Set "xpack.alerting.rules.minimumScheduleInterval.enforce" to true to prevent such changes.`);
  }
}

/**
 * Validate that updated schedule interval is not longer than any of the existing action frequencies
 * @param schedule Schedule interval that user tries to set
 * @param actions Rule actions
 */
function validateScheduleOperation(schedule, actions, ruleId) {
  const scheduleInterval = (0, _parse_duration.parseDuration)(schedule.interval);
  const actionsWithInvalidThrottles = [];
  for (const action of actions) {
    var _action$frequency;
    // check for actions throttled shorter than the rule schedule
    if (((_action$frequency = action.frequency) === null || _action$frequency === void 0 ? void 0 : _action$frequency.notifyWhen) === _types.RuleNotifyWhen.THROTTLE && (0, _parse_duration.parseDuration)(action.frequency.throttle) < scheduleInterval) {
      actionsWithInvalidThrottles.push(action);
    }
  }
  if (actionsWithInvalidThrottles.length > 0) {
    throw Error(`Error updating rule with ID "${ruleId}": the interval ${schedule.interval} is longer than the action frequencies`);
  }
}
async function prepareApiKeys(context, rule, ruleType, apiKeysMap, attributes, hasUpdateApiKeyOperation, username) {
  const apiKeyAttributes = await (0, _lib2.createNewAPIKeySet)(context, {
    id: ruleType.id,
    ruleName: attributes.name,
    username,
    shouldUpdateApiKey: attributes.enabled || hasUpdateApiKeyOperation,
    errorMessage: 'Error updating rule: could not create API key'
  });

  // collect generated API keys
  if (apiKeyAttributes.apiKey) {
    apiKeysMap.set(rule.id, {
      ...apiKeysMap.get(rule.id),
      newApiKey: apiKeyAttributes.apiKey,
      newApiKeyCreatedByUser: apiKeyAttributes.apiKeyCreatedByUser
    });
  }
  return {
    apiKeyAttributes
  };
}
function updateAttributes(context, attributes, apiKeyAttributes, updatedParams, rawAlertActions, username) {
  var _attributes$notifyWhe, _attributes$throttle;
  // get notifyWhen
  const notifyWhen = (0, _lib.getRuleNotifyWhenType)((_attributes$notifyWhe = attributes.notifyWhen) !== null && _attributes$notifyWhe !== void 0 ? _attributes$notifyWhe : null, (_attributes$throttle = attributes.throttle) !== null && _attributes$throttle !== void 0 ? _attributes$throttle : null);
  const updatedAttributes = (0, _lib2.updateMeta)(context, {
    ...attributes,
    ...apiKeyAttributes,
    params: updatedParams,
    actions: rawAlertActions,
    notifyWhen,
    updatedBy: username,
    updatedAt: new Date().toISOString()
  });

  // add mapped_params
  const mappedParams = (0, _mapped_params_utils.getMappedParams)(updatedParams);
  if (Object.keys(mappedParams).length) {
    updatedAttributes.mapped_params = mappedParams;
  }
  return {
    updatedAttributes
  };
}
async function saveBulkUpdatedRules(context, rules, apiKeysMap) {
  const apiKeysToInvalidate = [];
  let result;
  try {
    result = await context.unsecuredSavedObjectsClient.bulkCreate(rules, {
      overwrite: true
    });
  } catch (e) {
    // avoid unused newly generated API keys
    if (apiKeysMap.size > 0) {
      await (0, _bulk_mark_api_keys_for_invalidation.bulkMarkApiKeysForInvalidation)({
        apiKeys: Array.from(apiKeysMap.values()).filter(value => value.newApiKey && !value.newApiKeyCreatedByUser).map(value => value.newApiKey)
      }, context.logger, context.unsecuredSavedObjectsClient);
    }
    throw e;
  }
  result.saved_objects.map(({
    id,
    error
  }) => {
    var _apiKeysMap$get, _apiKeysMap$get2, _apiKeysMap$get3, _apiKeysMap$get4;
    const oldApiKey = (_apiKeysMap$get = apiKeysMap.get(id)) === null || _apiKeysMap$get === void 0 ? void 0 : _apiKeysMap$get.oldApiKey;
    const oldApiKeyCreatedByUser = (_apiKeysMap$get2 = apiKeysMap.get(id)) === null || _apiKeysMap$get2 === void 0 ? void 0 : _apiKeysMap$get2.oldApiKeyCreatedByUser;
    const newApiKey = (_apiKeysMap$get3 = apiKeysMap.get(id)) === null || _apiKeysMap$get3 === void 0 ? void 0 : _apiKeysMap$get3.newApiKey;
    const newApiKeyCreatedByUser = (_apiKeysMap$get4 = apiKeysMap.get(id)) === null || _apiKeysMap$get4 === void 0 ? void 0 : _apiKeysMap$get4.newApiKeyCreatedByUser;

    // if SO wasn't saved and has new API key it will be invalidated
    if (error && newApiKey && !newApiKeyCreatedByUser) {
      apiKeysToInvalidate.push(newApiKey);
      // if SO saved and has old Api Key it will be invalidate
    } else if (!error && oldApiKey && !oldApiKeyCreatedByUser) {
      apiKeysToInvalidate.push(oldApiKey);
    }
  });
  return {
    result,
    apiKeysToInvalidate
  };
}
async function attemptToMigrateLegacyFrequency(context, operation, attributes, ruleType) {
  if (operation.field !== 'actions') throw new Error('Can only perform frequency migration on an action operation');
  // Try to remove the rule-level frequency params, and then validate actions
  if (typeof attributes.notifyWhen !== 'undefined') attributes.notifyWhen = undefined;
  if (attributes.throttle) attributes.throttle = undefined;
  await (0, _lib2.validateActions)(context, ruleType, {
    ...attributes,
    actions: operation.value
  });
  return attributes;
}