"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkAuthorizationAndGetTotal = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _apmUtils = require("@kbn/apm-utils");
var _authorization = require("../../authorization");
var _constants = require("../common/constants");
var _audit_events = require("../common/audit_events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const checkAuthorizationAndGetTotal = async (context, {
  filter,
  action
}) => {
  const actionToConstantsMapping = {
    DELETE: {
      WriteOperation: _authorization.WriteOperations.BulkDelete,
      RuleAuditAction: _audit_events.RuleAuditAction.DELETE
    },
    ENABLE: {
      WriteOperation: _authorization.WriteOperations.BulkEnable,
      RuleAuditAction: _audit_events.RuleAuditAction.ENABLE
    },
    DISABLE: {
      WriteOperation: _authorization.WriteOperations.BulkDisable,
      RuleAuditAction: _audit_events.RuleAuditAction.DISABLE
    }
  };
  const {
    aggregations,
    total
  } = await (0, _apmUtils.withSpan)({
    name: 'unsecuredSavedObjectsClient.find',
    type: 'rules'
  }, () => context.unsecuredSavedObjectsClient.find({
    filter,
    page: 1,
    perPage: 0,
    type: 'alert',
    aggs: {
      alertTypeId: {
        multi_terms: {
          terms: [{
            field: 'alert.attributes.alertTypeId'
          }, {
            field: 'alert.attributes.consumer'
          }]
        }
      }
    }
  }));
  if (total > _constants.MAX_RULES_NUMBER_FOR_BULK_OPERATION) {
    throw _boom.default.badRequest(`More than ${_constants.MAX_RULES_NUMBER_FOR_BULK_OPERATION} rules matched for bulk ${action.toLocaleLowerCase()}`);
  }
  const buckets = aggregations === null || aggregations === void 0 ? void 0 : aggregations.alertTypeId.buckets;
  if (buckets === undefined || (buckets === null || buckets === void 0 ? void 0 : buckets.length) === 0) {
    throw _boom.default.badRequest(`No rules found for bulk ${action.toLocaleLowerCase()}`);
  }
  await (0, _apmUtils.withSpan)({
    name: 'authorization.ensureAuthorized',
    type: 'rules'
  }, () => (0, _pMap.default)(buckets, async ({
    key: [ruleType, consumer, actions]
  }) => {
    context.ruleTypeRegistry.ensureRuleTypeEnabled(ruleType);
    try {
      await context.authorization.ensureAuthorized({
        ruleTypeId: ruleType,
        consumer,
        operation: actionToConstantsMapping[action].WriteOperation,
        entity: _authorization.AlertingAuthorizationEntity.Rule
      });
    } catch (error) {
      var _context$auditLogger;
      (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
        action: actionToConstantsMapping[action].RuleAuditAction,
        error
      }));
      throw error;
    }
  }, {
    concurrency: _constants.RULE_TYPE_CHECKS_CONCURRENCY
  }));
  return {
    total
  };
};
exports.checkAuthorizationAndGetTotal = checkAuthorizationAndGetTotal;