"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createResourceInstallationHelper = createResourceInstallationHelper;
exports.successResult = exports.errorResult = void 0;
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Helper function that queues up resources to initialize until we are
 * ready to begin initialization. Once we're ready, we start taking from
 * the queue and kicking off initialization.
 *
 * If a resource is added after we begin initialization, we push it onto
 * the queue and the running loop will handle it
 *
 * If a resource is added to the queue when the processing loop is not
 * running, kick off the processing loop
 */
function createResourceInstallationHelper(logger, commonResourcesInitPromise, installFn) {
  const initializedContexts = new Map();
  const waitUntilContextResourcesInstalled = async (context, namespace = _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING, timeoutMs) => {
    try {
      const {
        result: commonInitResult,
        error: commonInitError
      } = await commonResourcesInitPromise;
      if (commonInitResult) {
        await installFn(context, namespace, timeoutMs);
        return successResult();
      } else {
        logger.warn(`Common resources were not initialized, cannot initialize context for ${context.context}`);
        return errorResult(commonInitError);
      }
    } catch (err) {
      logger.error(`Error initializing context ${context.context} - ${err.message}`);
      return errorResult(err.message);
    }
  };
  return {
    add: (context, namespace = _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING, timeoutMs) => {
      initializedContexts.set(`${context.context}_${namespace}`,
      // Return a promise than can be checked when needed
      waitUntilContextResourcesInstalled(context, namespace, timeoutMs));
    },
    getInitializedContext: async (context, namespace) => {
      const key = `${context}_${namespace}`;
      return initializedContexts.has(key) ? initializedContexts.get(key) : errorResult(`Unrecognized context ${key}`);
    }
  };
}
const successResult = () => ({
  result: true
});
exports.successResult = successResult;
const errorResult = error => ({
  result: false,
  error
});
exports.errorResult = errorResult;