"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFetchStream = useFetchStream;
var _react = require("react");
var _fetch_stream = require("./fetch_stream");
var _string_reducer = require("./string_reducer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Custom hook to receive streaming data.
 *
 * @param endpoint - API endpoint including Kibana base path.
 * @param body - API request body.
 * @param options - Optional custom reducer and initial state.
 * @returns An object with streaming data and methods act on the stream.
 */
function useFetchStream(endpoint, body, options) {
  var _options$reducer, _options$initialState;
  const [errors, setErrors] = (0, _react.useState)([]);
  const [isCancelled, setIsCancelled] = (0, _react.useState)(false);
  const [isRunning, setIsRunning] = (0, _react.useState)(false);
  const reducer = (_options$reducer = options === null || options === void 0 ? void 0 : options.reducer) !== null && _options$reducer !== void 0 ? _options$reducer : _string_reducer.stringReducer;
  const initialState = (_options$initialState = options === null || options === void 0 ? void 0 : options.initialState) !== null && _options$initialState !== void 0 ? _options$initialState : '';
  const [data, dispatch] = (0, _react.useReducer)(reducer, initialState);
  const abortCtrl = (0, _react.useRef)(new AbortController());
  const addError = error => {
    setErrors(prevErrors => [...prevErrors, error]);
  };
  const start = async () => {
    if (isRunning) {
      addError('Restart not supported yet.');
      return;
    }
    setErrors([]);
    setIsRunning(true);
    setIsCancelled(false);
    abortCtrl.current = new AbortController();
    for await (const [fetchStreamError, actions] of (0, _fetch_stream.fetchStream)(endpoint, abortCtrl, body, options !== undefined)) {
      if (fetchStreamError !== null) {
        addError(fetchStreamError);
      } else if (actions.length > 0) {
        dispatch(actions);
      }
    }
    setIsRunning(false);
  };
  const cancel = () => {
    abortCtrl.current.abort();
    setIsCancelled(true);
    setIsRunning(false);
  };

  // If components using this custom hook get unmounted, cancel any ongoing request.
  (0, _react.useEffect)(() => {
    return () => abortCtrl.current.abort();
  }, []);
  return {
    cancel,
    data,
    dispatch,
    errors,
    isCancelled,
    isRunning,
    start
  };
}