"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.panOngoing = exports.touchOngoing = exports.resetTouchZoom = exports.startTouchZoom = exports.doPanFromPosition = exports.endDrag = exports.markDragStartPosition = exports.getFocusDomain = exports.kineticFlywheel = exports.doZoomAroundPosition = exports.doZoomFromJumpDelta = exports.doPanFromJumpDelta = exports.multiplierToZoom = exports.initialZoomPan = void 0;
var math_1 = require("../utils/math");
var VELOCITY_THRESHOLD = 0.01;
var DRAG_VELOCITY_ATTENUATION = 0.92;
var INITIAL_ZOOM = 5.248;
var INITIAL_PAN = 0.961;
var initialZoomPan = function () { return ({
    focus: { zoom: INITIAL_ZOOM, pan: INITIAL_PAN },
    flyVelocity: NaN,
    focusStart: { zoom: NaN, pan: NaN },
    dragStartPosition: NaN,
    lastDragPosition: NaN,
    dragVelocity: NaN,
}); };
exports.initialZoomPan = initialZoomPan;
var ZOOM_MIN = 0;
var ZOOM_MAX = 35;
var PAN_MIN = 0;
var PAN_MAX = 1;
var zoomToMultiplier = function (zoom) { return 1 / Math.pow(2, zoom); };
var multiplierToZoom = function (multiplier) { return Math.log2(1 / multiplier); };
exports.multiplierToZoom = multiplierToZoom;
var getFocusDomainRatio = function (zoom) { return 1 / (1 - zoomToMultiplier(zoom)) - 1; };
var clampZoom = function (zoom) { return (0, math_1.clamp)(zoom, ZOOM_MIN, ZOOM_MAX) || 0; };
var clampPan = function (pan) { return (0, math_1.clamp)(pan, PAN_MIN, PAN_MAX) || 0; };
var setPan = function (focus, pan) { return (focus.pan = pan); };
var setZoomPan = function (focus, newZoomPan) { return Object.assign(focus, newZoomPan); };
var getFocusForNewZoom = function (focus, pointerUnitLocation, newZoom, panDelta) {
    var oldInvisibleFraction = 1 - zoomToMultiplier(focus.zoom);
    var zoom = clampZoom(newZoom);
    var newInvisibleFraction = 1 - zoomToMultiplier(zoom);
    var requestedPan = (0, math_1.mix)(pointerUnitLocation + panDelta, focus.pan, oldInvisibleFraction / newInvisibleFraction);
    var pan = clampPan(requestedPan);
    return { zoom: zoom, pan: pan };
};
var getPanFromDelta = function (focus, panStart, relativeDelta) {
    var panDeltaPerDrag = getFocusDomainRatio(focus.zoom);
    var panDifference = panDeltaPerDrag * relativeDelta;
    return clampPan(panStart - panDifference);
};
var doPanFromJumpDelta = function (zoomPan, normalizedDeltaPan) {
    var deltaPan = zoomToMultiplier(zoomPan.focus.zoom) * normalizedDeltaPan;
    var newPan = clampPan(zoomPan.focus.pan + deltaPan);
    setPan(zoomPan.focus, newPan);
};
exports.doPanFromJumpDelta = doPanFromJumpDelta;
var doZoomFromJumpDelta = function (zoomPan, normalizedDeltaZoom, center) {
    var newZoom = zoomPan.focus.zoom + normalizedDeltaZoom;
    var newZoomPan = getFocusForNewZoom(zoomPan.focus, center, newZoom, 0);
    setZoomPan(zoomPan.focus, newZoomPan);
};
exports.doZoomFromJumpDelta = doZoomFromJumpDelta;
var doZoomAroundPosition = function (zoomPan, _a, center, zoomChange, panDelta, touch) {
    var cartesianWidth = _a.innerSize, cartesianLeft = _a.innerLeading;
    var unitZoomCenter = (0, math_1.unitClamp)((center - cartesianLeft) / cartesianWidth);
    var base = touch ? zoomPan.focusStart.zoom : zoomPan.focus.zoom;
    var newZoomPan = getFocusForNewZoom(zoomPan.focus, unitZoomCenter, base + zoomChange, panDelta);
    setZoomPan(zoomPan.focus, newZoomPan);
};
exports.doZoomAroundPosition = doZoomAroundPosition;
var kineticFlywheel = function (zoomPan, size) {
    var velocity = zoomPan.flyVelocity;
    var needsViewUpdate = Math.abs(velocity) > VELOCITY_THRESHOLD;
    if (needsViewUpdate) {
        var newPan = getPanFromDelta(zoomPan.focus, zoomPan.focus.pan, velocity / size);
        setPan(zoomPan.focus, newPan);
        zoomPan.flyVelocity *= DRAG_VELOCITY_ATTENUATION;
    }
    else {
        zoomPan.flyVelocity = NaN;
    }
    return needsViewUpdate;
};
exports.kineticFlywheel = kineticFlywheel;
var getFocusDomain = function (zoomPan, referenceDomainFrom, referenceDomainTo) {
    var _a = zoomPan.focus, zoom = _a.zoom, pan = _a.pan;
    var referenceDomainExtent = referenceDomainTo - referenceDomainFrom;
    var focusDomainExtent = referenceDomainExtent * zoomToMultiplier(zoom);
    var leeway = referenceDomainExtent - focusDomainExtent;
    var domainFrom = referenceDomainFrom + pan * leeway;
    var domainTo = referenceDomainTo - (1 - pan) * leeway;
    return { domainFrom: domainFrom, domainTo: domainTo };
};
exports.getFocusDomain = getFocusDomain;
var markDragStartPosition = function (zoomPan, position) {
    zoomPan.dragStartPosition = position;
    zoomPan.lastDragPosition = position;
    zoomPan.dragVelocity = NaN;
    zoomPan.flyVelocity = NaN;
    zoomPan.focusStart.pan = zoomPan.focus.pan;
};
exports.markDragStartPosition = markDragStartPosition;
var endDrag = function (zoomPan) {
    zoomPan.flyVelocity = zoomPan.dragVelocity;
    zoomPan.dragVelocity = NaN;
    zoomPan.dragStartPosition = NaN;
    zoomPan.focusStart.pan = NaN;
};
exports.endDrag = endDrag;
var doPanFromPosition = function (zoomPan, innerSize, currentPosition) {
    var positionDelta = currentPosition - zoomPan.lastDragPosition;
    var dragVelocity = zoomPan.dragVelocity;
    zoomPan.dragVelocity =
        positionDelta * dragVelocity > 0 && Math.abs(positionDelta) < Math.abs(dragVelocity)
            ? dragVelocity
            : positionDelta;
    zoomPan.lastDragPosition = currentPosition;
    var delta = currentPosition - zoomPan.dragStartPosition;
    var newPan = getPanFromDelta(zoomPan.focus, zoomPan.focusStart.pan, delta / innerSize);
    setPan(zoomPan.focus, newPan);
};
exports.doPanFromPosition = doPanFromPosition;
var startTouchZoom = function (zoomPan) { return (zoomPan.focusStart.zoom = zoomPan.focus.zoom); };
exports.startTouchZoom = startTouchZoom;
var resetTouchZoom = function (zoomPan) { return (zoomPan.focusStart.zoom = NaN); };
exports.resetTouchZoom = resetTouchZoom;
var touchOngoing = function (zoomPan) { return Number.isFinite(zoomPan.focusStart.pan); };
exports.touchOngoing = touchOngoing;
var panOngoing = function (zoomPan) { return Number.isFinite(zoomPan.dragStartPosition); };
exports.panOngoing = panOngoing;
//# sourceMappingURL=zoom_pan.js.map