"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SyntheticsPrivateLocation = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _format_synthetics_policy = require("../../../common/formatters/format_synthetics_policy");

var _private_locations = require("../../legacy_uptime/lib/saved_objects/private_locations");

var _runtime_types = require("../../../common/runtime_types");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
class SyntheticsPrivateLocation {
  constructor(_server) {
    (0, _defineProperty2.default)(this, "server", void 0);
    this.server = _server;
  }

  getSpaceId(request) {
    return this.server.spaces.spacesService.getSpaceId(request);
  }

  getPolicyId(config, {
    id: locId
  }, request) {
    if (config[_runtime_types.ConfigKey.MONITOR_SOURCE_TYPE] === _runtime_types.SourceType.PROJECT) {
      return `${config.id}-${locId}`;
    }

    return `${config.id}-${locId}-${this.getSpaceId(request)}`;
  }

  async generateNewPolicy(config, privateLocation, request, savedObjectsClient) {
    if (!savedObjectsClient) {
      throw new Error('Could not find savedObjectsClient');
    }

    const {
      label: locName
    } = privateLocation;
    const spaceId = this.getSpaceId(request);

    try {
      var _config$fields, _config$fields2, _config$fields3;

      const newPolicy = await this.server.fleet.packagePolicyService.buildPackagePolicyFromPackage(savedObjectsClient, 'synthetics', this.server.logger);

      if (!newPolicy) {
        throw new Error(`Unable to create Synthetics package policy for private location ${privateLocation.label}`);
      }

      newPolicy.is_managed = true;
      newPolicy.policy_id = privateLocation.agentPolicyId;

      if (config[_runtime_types.ConfigKey.MONITOR_SOURCE_TYPE] === _runtime_types.SourceType.PROJECT) {
        newPolicy.name = `${config.id}-${locName}`;
      } else {
        newPolicy.name = `${config[_runtime_types.ConfigKey.NAME]}-${locName}-${spaceId}`;
      }

      newPolicy.output_id = '';
      newPolicy.namespace = 'default';
      const {
        formattedPolicy
      } = (0, _format_synthetics_policy.formatSyntheticsPolicy)(newPolicy, config.type, { ...config,
        config_id: (_config$fields = config.fields) === null || _config$fields === void 0 ? void 0 : _config$fields.config_id,
        location_name: privateLocation.label,
        'monitor.project.id': (_config$fields2 = config.fields) === null || _config$fields2 === void 0 ? void 0 : _config$fields2['monitor.project.name'],
        'monitor.project.name': (_config$fields3 = config.fields) === null || _config$fields3 === void 0 ? void 0 : _config$fields3['monitor.project.name']
      });
      return formattedPolicy;
    } catch (e) {
      this.server.logger.error(e);
      return null;
    }
  }

  async checkPermissions(request, error) {
    const {
      integrations: {
        writeIntegrationPolicies
      }
    } = await this.server.fleet.authz.fromRequest(request);

    if (!writeIntegrationPolicies) {
      throw new Error(error);
    }
  }

  async createMonitor(config, request, savedObjectsClient) {
    const {
      locations
    } = config;
    await this.checkPermissions(request, `Unable to create Synthetics package policy for monitor ${config[_runtime_types.ConfigKey.NAME]}. Fleet write permissions are needed to use Synthetics private locations.`);
    const privateLocations = await (0, _private_locations.getSyntheticsPrivateLocations)(savedObjectsClient);
    const fleetManagedLocations = locations.filter(loc => !loc.isServiceManaged);

    for (const privateLocation of fleetManagedLocations) {
      const location = privateLocations === null || privateLocations === void 0 ? void 0 : privateLocations.find(loc => loc.id === privateLocation.id);

      if (!location) {
        throw new Error(`Unable to find Synthetics private location for agentId ${privateLocation.id}`);
      }

      const newPolicy = await this.generateNewPolicy(config, location, request, savedObjectsClient);

      if (!newPolicy) {
        throw new Error(`Unable to create Synthetics package policy for monitor ${config[_runtime_types.ConfigKey.NAME]} with private location ${location.label}`);
      }

      try {
        await this.createPolicy(newPolicy, this.getPolicyId(config, location, request), savedObjectsClient);
      } catch (e) {
        this.server.logger.error(e);
        throw new Error(`Unable to create Synthetics package policy for monitor ${config[_runtime_types.ConfigKey.NAME]} with private location ${location.label}`);
      }
    }
  }

  async editMonitor(config, request, savedObjectsClient) {
    await this.checkPermissions(request, `Unable to update Synthetics package policy for monitor ${config[_runtime_types.ConfigKey.NAME]}. Fleet write permissions are needed to use Synthetics private locations.`);
    const {
      locations
    } = config;
    const allPrivateLocations = await (0, _private_locations.getSyntheticsPrivateLocations)(savedObjectsClient);
    const monitorPrivateLocations = locations.filter(loc => !loc.isServiceManaged);

    for (const privateLocation of allPrivateLocations) {
      const hasLocation = monitorPrivateLocations === null || monitorPrivateLocations === void 0 ? void 0 : monitorPrivateLocations.some(loc => loc.id === privateLocation.id);
      const currId = this.getPolicyId(config, privateLocation, request);
      const hasPolicy = await this.getMonitor(currId, savedObjectsClient);

      try {
        if (hasLocation) {
          const newPolicy = await this.generateNewPolicy(config, privateLocation, request, savedObjectsClient);

          if (!newPolicy) {
            throw new Error(`Unable to ${hasPolicy ? 'update' : 'create'} Synthetics package policy for private location ${privateLocation.label}`);
          }

          if (hasPolicy) {
            await this.updatePolicy(newPolicy, currId, savedObjectsClient);
          } else {
            await this.createPolicy(newPolicy, currId, savedObjectsClient);
          }
        } else if (hasPolicy) {
          const soClient = savedObjectsClient;
          const esClient = this.server.uptimeEsClient.baseESClient;

          try {
            await this.server.fleet.packagePolicyService.delete(soClient, esClient, [currId], {
              force: true
            });
          } catch (e) {
            this.server.logger.error(e);
            throw new Error(`Unable to delete Synthetics package policy for monitor ${config[_runtime_types.ConfigKey.NAME]} with private location ${privateLocation.label}`);
          }
        }
      } catch (e) {
        this.server.logger.error(e);
        throw new Error(`Unable to ${hasPolicy ? 'update' : 'create'} Synthetics package policy for monitor ${config[_runtime_types.ConfigKey.NAME]} with private location ${privateLocation.label}`);
      }
    }
  }

  async createPolicy(newPolicy, id, savedObjectsClient) {
    const soClient = savedObjectsClient;
    const esClient = this.server.uptimeEsClient.baseESClient;

    if (soClient && esClient) {
      return await this.server.fleet.packagePolicyService.create(soClient, esClient, newPolicy, {
        id,
        overwrite: true
      });
    }
  }

  async updatePolicy(updatedPolicy, id, savedObjectsClient) {
    const soClient = savedObjectsClient;
    const esClient = this.server.uptimeEsClient.baseESClient;

    if (soClient && esClient) {
      return await this.server.fleet.packagePolicyService.update(soClient, esClient, id, updatedPolicy, {
        force: true
      });
    }
  }

  async getMonitor(id, savedObjectsClient) {
    try {
      const soClient = savedObjectsClient;
      return await this.server.fleet.packagePolicyService.get(soClient, id);
    } catch (e) {
      this.server.logger.debug(e);
      return null;
    }
  }

  async deleteMonitor(config, request, savedObjectsClient) {
    const soClient = savedObjectsClient;
    const esClient = this.server.uptimeEsClient.baseESClient;

    if (soClient && esClient) {
      const {
        locations
      } = config;
      const allPrivateLocations = await (0, _private_locations.getSyntheticsPrivateLocations)(soClient);
      const monitorPrivateLocations = locations.filter(loc => !loc.isServiceManaged);

      for (const privateLocation of monitorPrivateLocations) {
        const location = allPrivateLocations === null || allPrivateLocations === void 0 ? void 0 : allPrivateLocations.find(loc => loc.id === privateLocation.id);

        if (location) {
          await this.checkPermissions(request, `Unable to delete Synthetics package policy for monitor ${config[_runtime_types.ConfigKey.NAME]}. Fleet write permissions are needed to use Synthetics private locations.`);

          try {
            await this.server.fleet.packagePolicyService.delete(soClient, esClient, [this.getPolicyId(config, location, request)], {
              force: true
            });
          } catch (e) {
            this.server.logger.error(e);
            throw new Error(`Unable to delete Synthetics package policy for monitor ${config[_runtime_types.ConfigKey.NAME]} with private location ${location.label}`);
          }
        }
      }
    }
  }

  async getAgentPolicies() {
    const agentPolicies = await this.server.fleet.agentPolicyService.list(this.server.savedObjectsClient, {
      page: 1,
      perPage: 10000
    });
    return agentPolicies.items;
  }

}

exports.SyntheticsPrivateLocation = SyntheticsPrivateLocation;