"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sleep = exports.sendFleetActionResponse = exports.sendEndpointActionResponse = exports.fleetActionGenerator = exports.fetchEndpointActionList = exports.endpointActionGenerator = void 0;

var _common = require("../../../../fleet/common");

var _endpoint_metadata_services = require("../common/endpoint_metadata_services");

var _fleet_action_generator = require("../../../common/endpoint/data_generators/fleet_action_generator");

var _constants = require("../../../common/endpoint/constants");

var _endpoint_action_generator = require("../../../common/endpoint/data_generators/endpoint_action_generator");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const ES_INDEX_OPTIONS = {
  headers: {
    'X-elastic-product-origin': 'fleet'
  }
};
const fleetActionGenerator = new _fleet_action_generator.FleetActionGenerator();
exports.fleetActionGenerator = fleetActionGenerator;
const endpointActionGenerator = new _endpoint_action_generator.EndpointActionGenerator();
exports.endpointActionGenerator = endpointActionGenerator;

const sleep = (ms = 1000) => new Promise(r => setTimeout(r, ms));

exports.sleep = sleep;

const fetchEndpointActionList = async (kbn, options = {}) => {
  return (await kbn.request({
    method: 'GET',
    path: _constants.ENDPOINTS_ACTION_LIST_ROUTE,
    query: options
  })).data;
};

exports.fetchEndpointActionList = fetchEndpointActionList;

const sendFleetActionResponse = async (esClient, action, {
  state
} = {}) => {
  const fleetResponse = fleetActionGenerator.generateResponse({
    action_id: action.id,
    agent_id: action.agents[0],
    action_response: {
      endpoint: {
        ack: true
      }
    }
  }); // 20% of the time we generate an error

  if (state === 'failure' || !state && fleetActionGenerator.randomFloat() < 0.2) {
    fleetResponse.action_response = {};
    fleetResponse.error = 'Agent failed to deliver message to endpoint due to unknown error';
  } else {
    // show it as success (generator currently always generates a `error`, so delete it)
    delete fleetResponse.error;
  }

  await esClient.index({
    index: _common.AGENT_ACTIONS_RESULTS_INDEX,
    body: fleetResponse,
    refresh: 'wait_for'
  }, ES_INDEX_OPTIONS);
  return fleetResponse;
};

exports.sendFleetActionResponse = sendFleetActionResponse;

const sendEndpointActionResponse = async (esClient, action, {
  state
} = {}) => {
  const endpointResponse = endpointActionGenerator.generateResponse({
    agent: {
      id: action.agents[0]
    },
    EndpointActions: {
      action_id: action.id,
      data: {
        command: action.command,
        comment: '',
        ...getOutputDataIfNeeded(action.command)
      },
      started_at: action.startedAt
    }
  }); // 20% of the time we generate an error

  if (state === 'failure' || state !== 'success' && endpointActionGenerator.randomFloat() < 0.2) {
    endpointResponse.error = {
      message: 'Endpoint encountered an error and was unable to apply action to host'
    };
  }

  await esClient.index({
    index: _constants.ENDPOINT_ACTION_RESPONSES_INDEX,
    body: endpointResponse,
    refresh: 'wait_for'
  }); // ------------------------------------------
  // Post Action Response tasks
  // ------------------------------------------
  // For isolate, If the response is not an error, then also send a metadata update

  if (action.command === 'isolate' && !endpointResponse.error) {
    for (const agentId of action.agents) {
      await (0, _endpoint_metadata_services.sendEndpointMetadataUpdate)(esClient, agentId, {
        Endpoint: {
          state: {
            isolation: true
          }
        }
      });
    }
  } // For UnIsolate, if response is not an Error, then also send metadata update


  if (action.command === 'unisolate' && !endpointResponse.error) {
    for (const agentId of action.agents) {
      await (0, _endpoint_metadata_services.sendEndpointMetadataUpdate)(esClient, agentId, {
        Endpoint: {
          state: {
            isolation: false
          }
        }
      });
    }
  }

  return endpointResponse;
};

exports.sendEndpointActionResponse = sendEndpointActionResponse;

const getOutputDataIfNeeded = command => {
  return command === 'running-processes' ? {
    output: {
      type: 'json',
      content: {
        entries: endpointActionGenerator.randomResponseActionProcesses(100)
      }
    }
  } : {};
};