"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDockerRegistry = useDockerRegistry;

var Rx = _interopRequireWildcard(require("rxjs"));

var _operators = require("rxjs/operators");

var _execa = _interopRequireDefault(require("execa"));

var _stdioDevHelpers = require("@kbn/stdio-dev-helpers");

var _toolingLog = require("@kbn/tooling-log");

var _pRetry = _interopRequireDefault(require("p-retry"));

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const BEFORE_SETUP_TIMEOUT = 30 * 60 * 1000; // 30 minutes;

const DOCKER_START_TIMEOUT = 5 * 60 * 1000; // 5 minutes

const DOCKER_IMAGE = `docker.elastic.co/package-registry/distribution:production-v2-experimental`;

function firstWithTimeout(source$, errorMsg, ms = 30 * 1000) {
  return Rx.race(source$.pipe((0, _operators.take)(1)), Rx.timer(ms).pipe((0, _operators.map)(() => {
    throw new Error(`[docker:${DOCKER_IMAGE}] ${errorMsg} within ${ms / 1000} seconds`);
  })));
}

function childProcessToLogLine(childProcess, log) {
  const logLine$ = new Rx.Subject();
  Rx.merge((0, _stdioDevHelpers.observeLines)(childProcess.stdout).pipe((0, _operators.tap)(line => log.info(`[docker:${DOCKER_IMAGE}] ${line}`))), // TypeScript note: As long as the proc stdio[1] is 'pipe', then stdout will not be null
  (0, _stdioDevHelpers.observeLines)(childProcess.stderr).pipe((0, _operators.tap)(line => log.error(`[docker:${DOCKER_IMAGE}] ${line}`))) // TypeScript note: As long as the proc stdio[2] is 'pipe', then stderr will not be null
  ).subscribe(logLine$);
  return logLine$.asObservable();
}

function useDockerRegistry() {
  const logger = new _toolingLog.ToolingLog({
    level: 'info',
    writeTo: process.stdout
  });
  const packageRegistryPort = process.env.FLEET_PACKAGE_REGISTRY_PORT || '8081';

  if (!packageRegistryPort.match(/^[0-9]{4}/)) {
    throw new Error('Invalid FLEET_PACKAGE_REGISTRY_PORT');
  }

  let dockerProcess;

  async function startDockerRegistryServer() {
    const args = ['run', '--rm', '-p', `${packageRegistryPort}:8080`, DOCKER_IMAGE];
    dockerProcess = (0, _execa.default)('docker', args, {
      stdio: ['ignore', 'pipe', 'pipe']
    });
    let isExited = dockerProcess.exitCode !== null;
    dockerProcess.once('exit', () => {
      isExited = true;
    });
    const waitForLogLine = /package manifests loaded/;

    try {
      await firstWithTimeout(childProcessToLogLine(dockerProcess, logger).pipe((0, _operators.filter)(line => {
        process.stdout.write(line);
        return waitForLogLine.test(line);
      })), 'no package manifests loaded', DOCKER_START_TIMEOUT).toPromise();
    } catch (err) {
      dockerProcess.kill();
      throw err;
    }

    if (isExited && dockerProcess.exitCode !== 0) {
      throw new Error(`Unable to setup docker registry exit code ${dockerProcess.exitCode}`);
    }
  }

  async function pullDockerImage() {
    logger.info(`[docker:${DOCKER_IMAGE}] pulling docker image "${DOCKER_IMAGE}"`);
    await (0, _execa.default)('docker', ['pull', DOCKER_IMAGE]);
  }

  async function cleanupDockerRegistryServer() {
    if (dockerProcess && !dockerProcess.killed) {
      dockerProcess.kill();
    }
  }

  beforeAll(async () => {
    jest.setTimeout(BEFORE_SETUP_TIMEOUT);
    await (0, _pRetry.default)(() => pullDockerImage(), {
      retries: 3
    });
    await (0, _pRetry.default)(() => startDockerRegistryServer(), {
      retries: 3
    });
  });
  afterAll(async () => {
    await cleanupDockerRegistryServer();
  });
  return `http://localhost:${packageRegistryPort}`;
}