"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerIndexRoutes = registerIndexRoutes;

var _configSchema = require("@kbn/config-schema");

var _i18n = require("@kbn/i18n");

var _error_codes = require("../../../common/types/error_codes");

var _fetch_connectors = require("../../lib/connectors/fetch_connectors");

var _fetch_crawlers = require("../../lib/crawler/fetch_crawlers");

var _create_index = require("../../lib/indices/create_index");

var _fetch_index = require("../../lib/indices/fetch_index");

var _fetch_indices = require("../../lib/indices/fetch_indices");

var _generate_api_key = require("../../lib/indices/generate_api_key");

var _create_error = require("../../utils/create_error");

var _elasticsearch_error_handler = require("../../utils/elasticsearch_error_handler");

var _identify_exceptions = require("../../utils/identify_exceptions");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
function registerIndexRoutes({
  router,
  log
}) {
  router.get({
    path: '/internal/enterprise_search/search_indices',
    validate: false
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, _, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const indices = await (0, _fetch_indices.fetchIndices)(client, '*', false, true, 'search-');
    return response.ok({
      body: indices,
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.get({
    path: '/internal/enterprise_search/indices',
    validate: {
      query: _configSchema.schema.object({
        page: _configSchema.schema.number({
          defaultValue: 0,
          min: 0
        }),
        return_hidden_indices: _configSchema.schema.maybe(_configSchema.schema.boolean()),
        search_query: _configSchema.schema.maybe(_configSchema.schema.string()),
        size: _configSchema.schema.number({
          defaultValue: 10,
          min: 0
        })
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      page,
      size,
      return_hidden_indices: returnHiddenIndices,
      search_query: searchQuery
    } = request.query;
    const {
      client
    } = (await context.core).elasticsearch;
    const indexPattern = searchQuery ? `*${searchQuery}*` : '*';
    const totalIndices = await (0, _fetch_indices.fetchIndices)(client, indexPattern, !!returnHiddenIndices, false);
    const totalResults = totalIndices.length;
    const totalPages = Math.ceil(totalResults / size) || 1;
    const startIndex = (page - 1) * size;
    const endIndex = page * size;
    const selectedIndices = totalIndices.slice(startIndex, endIndex);
    const indexNames = selectedIndices.map(({
      name
    }) => name);
    const connectors = await (0, _fetch_connectors.fetchConnectors)(client, indexNames);
    const crawlers = await (0, _fetch_crawlers.fetchCrawlers)(client, indexNames);
    const indices = selectedIndices.map(index => ({ ...index,
      connector: connectors.find(connector => connector.index_name === index.name),
      crawler: crawlers.find(crawler => crawler.index_name === index.name)
    }));
    return response.ok({
      body: {
        indices,
        meta: {
          page: {
            current: page,
            size: indices.length,
            total_pages: totalPages,
            total_results: totalResults
          }
        }
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.get({
    path: '/internal/enterprise_search/indices/{indexName}',
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const indexName = decodeURIComponent(request.params.indexName);
    const {
      client
    } = (await context.core).elasticsearch;

    try {
      const index = await (0, _fetch_index.fetchIndex)(client, indexName);
      return response.ok({
        body: index,
        headers: {
          'content-type': 'application/json'
        }
      });
    } catch (error) {
      if ((0, _identify_exceptions.isIndexNotFoundException)(error)) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.INDEX_NOT_FOUND,
          message: 'Could not find index',
          response,
          statusCode: 404
        });
      }

      throw error;
    }
  }));
  router.get({
    path: '/internal/enterprise_search/indices/{indexName}/exists',
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const indexName = decodeURIComponent(request.params.indexName);
    const {
      client
    } = (await context.core).elasticsearch;
    let indexExists;

    try {
      indexExists = await client.asCurrentUser.indices.exists({
        index: indexName
      });
    } catch (e) {
      log.warn(_i18n.i18n.translate('xpack.enterpriseSearch.server.routes.indices.existsErrorLogMessage', {
        defaultMessage: 'An error occured while resolving request to {requestUrl}',
        values: {
          requestUrl: request.url.toString()
        }
      }));
      log.warn(e);
      indexExists = false;
    }

    return response.ok({
      body: {
        exists: indexExists
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: '/internal/enterprise_search/indices/{indexName}/api_key',
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const indexName = decodeURIComponent(request.params.indexName);
    const {
      client
    } = (await context.core).elasticsearch;
    const apiKey = await (0, _generate_api_key.generateApiKey)(client, indexName);
    return response.ok({
      body: apiKey,
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: '/internal/enterprise_search/indices',
    validate: {
      body: _configSchema.schema.object({
        index_name: _configSchema.schema.string(),
        language: _configSchema.schema.maybe(_configSchema.schema.nullable(_configSchema.schema.string()))
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      ['index_name']: indexName,
      language
    } = request.body;
    const {
      client
    } = (await context.core).elasticsearch;
    const indexExists = await client.asCurrentUser.indices.exists({
      index: request.body.index_name
    });

    if (indexExists) {
      return (0, _create_error.createError)({
        errorCode: _error_codes.ErrorCode.INDEX_ALREADY_EXISTS,
        message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.createApiIndex.indexExistsError', {
          defaultMessage: 'This index already exists'
        }),
        response,
        statusCode: 409
      });
    }

    const crawler = await (0, _fetch_crawlers.fetchCrawlerByIndexName)(client, request.body.index_name);

    if (crawler) {
      return (0, _create_error.createError)({
        errorCode: _error_codes.ErrorCode.CRAWLER_ALREADY_EXISTS,
        message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.createApiIndex.crawlerExistsError', {
          defaultMessage: 'A crawler for this index already exists'
        }),
        response,
        statusCode: 409
      });
    }

    const connector = await (0, _fetch_connectors.fetchConnectorByIndexName)(client, request.body.index_name);

    if (connector) {
      return (0, _create_error.createError)({
        errorCode: _error_codes.ErrorCode.CONNECTOR_DOCUMENT_ALREADY_EXISTS,
        message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.createApiIndex.connectorExistsError', {
          defaultMessage: 'A connector for this index already exists'
        }),
        response,
        statusCode: 409
      });
    }

    const createIndexResponse = await (0, _create_index.createApiIndex)(client, indexName, language);
    return response.ok({
      body: createIndexResponse,
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
}