"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateRulesConfigurationBodySchema = exports.updateAgentConfiguration = exports.setVarToPackagePolicy = exports.getPackagePolicy = exports.getCspRules = exports.defineUpdateRulesConfigRoute = exports.createRulesConfig = exports.convertRulesConfigToYaml = void 0;

var _configSchema = require("@kbn/config-schema");

var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");

var _immer = require("immer");

var _lodash = require("lodash");

var _jsYaml = _interopRequireDefault(require("js-yaml"));

var _helpers = require("../../../common/utils/helpers");

var _constants = require("../../../common/constants");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const getPackagePolicy = async (soClient, packagePolicyService, packagePolicyId) => {
  var _packagePolicies$0$pa;

  const packagePolicies = await packagePolicyService.getByIDs(soClient, [packagePolicyId]); // PackagePolicies always contains one element, even when package does not exist

  if (!packagePolicies || !packagePolicies[0].version) {
    throw new Error(`package policy Id '${packagePolicyId}' is not exist`);
  }

  if (((_packagePolicies$0$pa = packagePolicies[0].package) === null || _packagePolicies$0$pa === void 0 ? void 0 : _packagePolicies$0$pa.name) !== _constants.CLOUD_SECURITY_POSTURE_PACKAGE_NAME) {
    throw new Error(`Package Policy Id '${packagePolicyId}' is not of type cloud security posture package`);
  }

  return packagePolicies[0];
};

exports.getPackagePolicy = getPackagePolicy;

const getCspRules = (soClient, packagePolicy) => {
  return soClient.find({
    type: _constants.CSP_RULE_SAVED_OBJECT_TYPE,
    filter: (0, _helpers.createCspRuleSearchFilterByPackagePolicy)({
      packagePolicyId: packagePolicy.id,
      policyId: packagePolicy.policy_id
    }),
    searchFields: ['name'],
    // TODO: research how to get all rules
    perPage: 10000
  });
};

exports.getCspRules = getCspRules;

const getEnabledRulesByBenchmark = rules => rules.reduce((benchmarks, rule) => {
  const benchmark = benchmarks[rule.attributes.metadata.benchmark.id];
  if (!rule.attributes.enabled || !benchmark) return benchmarks;
  benchmark.push(rule.attributes.metadata.rego_rule_id);
  return benchmarks;
}, {
  cis_k8s: [],
  cis_eks: []
});

const createRulesConfig = cspRules => ({
  data_yaml: {
    activated_rules: getEnabledRulesByBenchmark(cspRules.saved_objects)
  }
});

exports.createRulesConfig = createRulesConfig;

const convertRulesConfigToYaml = config => {
  return _jsYaml.default.safeDump(config);
};

exports.convertRulesConfigToYaml = convertRulesConfigToYaml;

const setVarToPackagePolicy = (packagePolicy, dataYaml) => {
  const configFile = {
    dataYaml: {
      type: 'yaml',
      value: dataYaml
    }
  };
  const updatedPackagePolicy = (0, _immer.produce)(packagePolicy, draft => {
    (0, _lodash.unset)(draft, 'id');

    if (draft.vars) {
      draft.vars.dataYaml = configFile.dataYaml;
    } else {
      draft.vars = configFile;
    }
  });
  return updatedPackagePolicy;
};

exports.setVarToPackagePolicy = setVarToPackagePolicy;

const updateAgentConfiguration = async (packagePolicyService, packagePolicy, esClient, soClient, user) => {
  const cspRules = await getCspRules(soClient, packagePolicy);
  const rulesConfig = createRulesConfig(cspRules);
  const dataYaml = convertRulesConfigToYaml(rulesConfig);
  const updatedPackagePolicy = setVarToPackagePolicy(packagePolicy, dataYaml);
  const options = {
    user: user ? user : undefined
  };
  return packagePolicyService.update(soClient, esClient, packagePolicy.id, updatedPackagePolicy, options);
};

exports.updateAgentConfiguration = updateAgentConfiguration;

const defineUpdateRulesConfigRoute = router => router.post({
  path: _constants.UPDATE_RULES_CONFIG_ROUTE_PATH,
  validate: {
    body: updateRulesConfigurationBodySchema
  },
  options: {
    tags: ['access:cloud-security-posture-all']
  }
}, async (context, request, response) => {
  const cspContext = await context.csp;

  if (!(await context.fleet).authz.fleet.all) {
    return response.forbidden();
  }

  try {
    const packagePolicy = await getPackagePolicy(cspContext.soClient, cspContext.packagePolicyService, request.body.package_policy_id);
    const updatedPackagePolicy = await updateAgentConfiguration(cspContext.packagePolicyService, packagePolicy, cspContext.esClient.asCurrentUser, cspContext.soClient, cspContext.user);
    return response.ok({
      body: updatedPackagePolicy
    });
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    cspContext.logger.error(`Failed to update rules configuration on package policy ${error.message}`);
    return response.customError({
      body: {
        message: error.message
      },
      statusCode: error.statusCode
    });
  }
});

exports.defineUpdateRulesConfigRoute = defineUpdateRulesConfigRoute;

const updateRulesConfigurationBodySchema = _configSchema.schema.object({
  /**
   * CSP integration instance ID
   */
  package_policy_id: _configSchema.schema.string()
});

exports.updateRulesConfigurationBodySchema = updateRulesConfigurationBodySchema;