import _createForOfIteratorHelper from "@babel/runtime/helpers/createForOfIteratorHelper";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { i18n } from '@kbn/i18n';
export var FILENAME_WILDCARD_WARNING = i18n.translate('utils.filename.wildcardWarning', {
  defaultMessage: "Using wildcards in file paths can impact Endpoint performance"
});
export var FILEPATH_WARNING = i18n.translate('utils.filename.pathWarning', {
  defaultMessage: "Path may be formed incorrectly; verify value"
});
export var ConditionEntryField;

(function (ConditionEntryField) {
  ConditionEntryField["HASH"] = "process.hash.*";
  ConditionEntryField["PATH"] = "process.executable.caseless";
  ConditionEntryField["SIGNER"] = "process.Ext.code_signature";
})(ConditionEntryField || (ConditionEntryField = {}));

export var EntryFieldType;

(function (EntryFieldType) {
  EntryFieldType["HASH"] = ".hash.";
  EntryFieldType["EXECUTABLE"] = ".executable.caseless";
  EntryFieldType["PATH"] = ".path";
  EntryFieldType["SIGNER"] = ".Ext.code_signature";
})(EntryFieldType || (EntryFieldType = {}));

export var OperatingSystem;

(function (OperatingSystem) {
  OperatingSystem["LINUX"] = "linux";
  OperatingSystem["MAC"] = "macos";
  OperatingSystem["WINDOWS"] = "windows";
})(OperatingSystem || (OperatingSystem = {}));

export var validateFilePathInput = function validateFilePathInput(_ref) {
  var os = _ref.os,
      _ref$value = _ref.value,
      value = _ref$value === void 0 ? '' : _ref$value;
  var textInput = value.trim();
  var isValidFilePath = isPathValid({
    os: os,
    field: 'file.path.text',
    type: 'wildcard',
    value: textInput
  });
  var hasSimpleFileName = hasSimpleExecutableName({
    os: os,
    type: 'wildcard',
    value: textInput
  });

  if (!textInput.length) {
    return FILEPATH_WARNING;
  }

  if (isValidFilePath) {
    if (hasSimpleFileName !== undefined && !hasSimpleFileName) {
      return FILENAME_WILDCARD_WARNING;
    }
  } else {
    return FILEPATH_WARNING;
  }
};
export var hasSimpleExecutableName = function hasSimpleExecutableName(_ref2) {
  var os = _ref2.os,
      type = _ref2.type,
      value = _ref2.value;
  var separator = os === OperatingSystem.WINDOWS ? '\\' : '/';
  var lastString = value.split(separator).pop();

  if (!lastString) {
    return;
  }

  if (type === 'wildcard') {
    return (lastString.split('*').length || lastString.split('?').length) === 1;
  }

  return true;
};
export var isPathValid = function isPathValid(_ref3) {
  var os = _ref3.os,
      field = _ref3.field,
      type = _ref3.type,
      value = _ref3.value;
  var pathFields = ['process.executable.caseless', 'file.path', 'file.path.text'];

  if (pathFields.includes(field)) {
    if (type === 'wildcard') {
      return os === OperatingSystem.WINDOWS ? isWindowsWildcardPathValid(value) : isLinuxMacWildcardPathValid(value);
    }

    return doesPathMatchRegex({
      value: value,
      os: os
    });
  }

  return true;
};

var doesPathMatchRegex = function doesPathMatchRegex(_ref4) {
  var os = _ref4.os,
      value = _ref4.value;

  if (os === OperatingSystem.WINDOWS) {
    var filePathRegex = /^[a-z]:(?:|\\\\[^<>:"'/\\|?*]+\\[^<>:"'/\\|?*]+|%\w+%|)[\\](?:[^<>:"'/\\|?*]+[\\/])*([^<>:"'/\\|?*])+$/i;
    return filePathRegex.test(value);
  }

  return /^(\/|(\/[\w\-]+)+|\/[\w\-]+\.[\w]+|(\/[\w-]+)+\/[\w\-]+\.[\w]+)$/i.test(value);
};

var isWindowsWildcardPathValid = function isWindowsWildcardPathValid(path) {
  var firstCharacter = path[0];
  var lastCharacter = path.slice(-1);
  var trimmedValue = path.trim();
  var hasSlash = /\//.test(trimmedValue);

  if (path.length === 0) {
    return false;
  } else if (hasSlash || trimmedValue.length !== path.length || firstCharacter === '^' || lastCharacter === '\\' || !hasWildcard({
    path: path,
    isWindowsPath: true
  })) {
    return false;
  } else {
    return true;
  }
};

var isLinuxMacWildcardPathValid = function isLinuxMacWildcardPathValid(path) {
  var firstCharacter = path[0];
  var lastCharacter = path.slice(-1);
  var trimmedValue = path.trim();

  if (path.length === 0) {
    return false;
  } else if (trimmedValue.length !== path.length || firstCharacter !== '/' || lastCharacter === '/' || path.length > 1024 === true || path.includes('//') === true || !hasWildcard({
    path: path,
    isWindowsPath: false
  })) {
    return false;
  } else {
    return true;
  }
};

var hasWildcard = function hasWildcard(_ref5) {
  var path = _ref5.path,
      isWindowsPath = _ref5.isWindowsPath;

  var _iterator = _createForOfIteratorHelper(path.split(isWindowsPath ? '\\' : '/')),
      _step;

  try {
    for (_iterator.s(); !(_step = _iterator.n()).done;) {
      var pathComponent = _step.value;

      if (/[\*|\?]+/.test(pathComponent) === true) {
        return true;
      }
    }
  } catch (err) {
    _iterator.e(err);
  } finally {
    _iterator.f();
  }

  return false;
};