import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import * as t from 'io-ts';
import { either } from 'fp-ts/lib/Either';
import { difference, isPlainObject, forEach } from 'lodash';
var tags = ['DictionaryType', 'IntersectionType', 'MergeType', 'InterfaceType', 'PartialType', 'ExactType', 'UnionType'];

function isParsableType(type) {
  return tags.includes(type._tag);
}

function getHandlingTypes(type, key, value) {
  if (!isParsableType(type)) {
    return [];
  }

  switch (type._tag) {
    case 'DictionaryType':
      return [type.codomain];

    case 'IntersectionType':
      return type.types.map(function (i) {
        return getHandlingTypes(i, key, value);
      }).flat();

    case 'MergeType':
      return type.types.map(function (i) {
        return getHandlingTypes(i, key, value);
      }).flat();

    case 'InterfaceType':
    case 'PartialType':
      return [type.props[key]];

    case 'ExactType':
      return getHandlingTypes(type.type, key, value);

    case 'UnionType':
      var matched = type.types.find(function (m) {
        return m.is(value);
      });
      return matched ? getHandlingTypes(matched, key, value) : [];
  }
}

function getHandledKeys(type, object) {
  var prefix = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : '';
  var keys = {
    handled: new Set(),
    all: new Set()
  };
  forEach(object, function (value, key) {
    var ownPrefix = prefix ? "".concat(prefix, ".").concat(key) : key;
    keys.all.add(ownPrefix);
    var handlingTypes = getHandlingTypes(type, key, object).filter(Boolean);

    if (handlingTypes.length) {
      keys.handled.add(ownPrefix);
    }

    if (isPlainObject(value)) {
      handlingTypes.forEach(function (i) {
        var nextKeys = getHandledKeys(i, value, ownPrefix);
        nextKeys.all.forEach(function (k) {
          return keys.all.add(k);
        });
        nextKeys.handled.forEach(function (k) {
          return keys.handled.add(k);
        });
      });
    }
  });
  return keys;
}

export function strictKeysRt(type) {
  return new t.Type(type.name, type.is, function (input, context) {
    return either.chain(type.validate(input, context), function (i) {
      var keys = getHandledKeys(type, input);
      var excessKeys = difference(_toConsumableArray(keys.all), _toConsumableArray(keys.handled));

      if (excessKeys.length) {
        return t.failure(i, context, "Excess keys are not allowed:\n".concat(excessKeys.join('\n')));
      }

      return t.success(i);
    });
  }, type.encode);
}