"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toColorStops = exports.sortColorRanges = exports.isLastItem = exports.getValueForContinuity = exports.calculateMaxStep = void 0;

var _palettes = require("../../../../palettes");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Check if item is last
 * @internal
 */
const isLastItem = accessor => accessor === 'end';
/**
 * Sort Color ranges array
 * @internal
 */


exports.isLastItem = isLastItem;

const sortColorRanges = colorRanges => {
  const lastRange = colorRanges[colorRanges.length - 1]; // we should add last end as new start because we should include it on sorting

  return [...colorRanges, {
    start: lastRange.end,
    color: lastRange.color,
    end: undefined
  }].sort(({
    start: startA
  }, {
    start: startB
  }) => Number(startA) - Number(startB)).reduce((sortedColorRange, newColorRange, i, array) => {
    // we should pick correct color for the last range.
    // If after sorting we don't change last value we should just take color in array order
    // In another case we should get the next one.
    let color = newColorRange.color;

    if (i === array.length - 2 && array[i + 1].start !== lastRange.end) {
      color = array[i + 1].color;
    }

    if (i !== array.length - 1) {
      sortedColorRange.push({
        color,
        start: newColorRange.start,
        end: array[i + 1].start
      });
    }

    return sortedColorRange;
  }, []);
};
/**
 * Calculate max step
 * @internal
 */


exports.sortColorRanges = sortColorRanges;

const calculateMaxStep = (stops, max) => {
  let step = 1;

  if (stops.length > 1) {
    const last = stops[stops.length - 1];
    const last2step = stops[stops.length - 1] - stops[stops.length - 2];

    if (last + last2step < max) {
      step = last2step;
    }
  }

  return (0, _palettes.roundValue)(step);
};
/**
 * Convert ColorRange to ColorStops
 * @internal
 */


exports.calculateMaxStep = calculateMaxStep;

const toColorStops = (colorRanges, continuity) => {
  const min = (0, _palettes.checkIsMinContinuity)(continuity) ? Number.NEGATIVE_INFINITY : colorRanges[0].start;
  const max = (0, _palettes.checkIsMaxContinuity)(continuity) ? Number.POSITIVE_INFINITY : colorRanges[colorRanges.length - 1].end;
  return {
    min,
    max,
    colorStops: colorRanges.map((colorRange, i) => ({
      color: colorRange.color,
      stop: i === 0 ? min : colorRange.start
    }))
  };
};
/**
 * Calculate right max or min value for new continuity
 */


exports.toColorStops = toColorStops;

const getValueForContinuity = (colorRanges, continuity, isLast, rangeType, dataBounds) => {
  const {
    max,
    min
  } = (0, _palettes.getDataMinMax)(rangeType, dataBounds);
  let value;

  if (isLast) {
    if ((0, _palettes.checkIsMaxContinuity)(continuity)) {
      value = Number.POSITIVE_INFINITY;
    } else {
      value = (0, _palettes.roundValue)(colorRanges[colorRanges.length - 1].start > max ? colorRanges[colorRanges.length - 1].start + 1 : max);
    }
  } else {
    if ((0, _palettes.checkIsMinContinuity)(continuity)) {
      value = Number.NEGATIVE_INFINITY;
    } else {
      value = (0, _palettes.roundValue)(colorRanges[0].end < min ? colorRanges[0].end - 1 : min);
    }
  }

  return value;
};

exports.getValueForContinuity = getValueForContinuity;