"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateColorRanges = exports.validateColorRange = exports.isAllColorRangesValid = exports.getErrorMessages = void 0;

var _i18n = require("@kbn/i18n");

var _utils = require("../utils");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/** @internal **/
const getErrorMessages = colorRangesValidity => {
  return Array.from(new Set(Object.values(colorRangesValidity).map(item => item.errors).flat().map(item => {
    switch (item) {
      case 'invalidColor':
      case 'invalidValue':
        return _i18n.i18n.translate('coloring.dynamicColoring.customPalette.invalidValueOrColor', {
          defaultMessage: 'At least one color range contains the wrong value or color'
        });

      case 'greaterThanMaxValue':
        return _i18n.i18n.translate('coloring.dynamicColoring.customPalette.invalidMaxValue', {
          defaultMessage: 'Maximum value must be greater than preceding values'
        });

      default:
        return '';
    }
  })));
};
/** @internal **/


exports.getErrorMessages = getErrorMessages;

const validateColorRange = (colorRange, accessor) => {
  const errors = [];

  if (Number.isNaN(colorRange[accessor])) {
    errors.push('invalidValue');
  }

  if (accessor === 'end') {
    if (colorRange.start > colorRange.end) {
      errors.push('greaterThanMaxValue');
    }
  } else if (!(0, _utils.isValidColor)(colorRange.color)) {
    errors.push('invalidColor');
  }

  return {
    isValid: !errors.length,
    errors
  };
};

exports.validateColorRange = validateColorRange;

const validateColorRanges = colorRanges => {
  const validations = colorRanges.reduce((acc, item, index) => ({ ...acc,
    [index]: validateColorRange(item, 'start')
  }), {});
  return { ...validations,
    last: validateColorRange(colorRanges[colorRanges.length - 1], 'end')
  };
};

exports.validateColorRanges = validateColorRanges;

const isAllColorRangesValid = colorRanges => {
  return Object.values(validateColorRanges(colorRanges)).every(colorRange => colorRange.isValid);
};

exports.isAllColorRangesValid = isAllColorRangesValid;