"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.schemaToIoTs = schemaToIoTs;

var t = _interopRequireWildcard(require("io-ts"));

var _excess = require("./excess");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Is it a tuple of t.Mixed?
 * @param schemas Array of io-ts schemas
 */
function isOneOfCandidate(schemas) {
  return schemas.length === 2;
}
/**
 * Converts each {@link SchemaValue} to the io-ts equivalent
 * @param value The {@link SchemaValue} to parse
 */


function schemaValueToIoTs(value) {
  // We need to check the pass_through type on top of everything
  if (value.type === 'pass_through') {
    return t.unknown;
  }

  if ('properties' in value) {
    const {
      DYNAMIC_KEY,
      ...properties
    } = value.properties;
    const schemas = [schemaObjectToIoTs({
      properties
    })];

    if (DYNAMIC_KEY) {
      schemas.push(t.record(t.string, schemaValueToIoTs(DYNAMIC_KEY)));
    }

    return isOneOfCandidate(schemas) ? t.union(schemas) : schemas[0];
  } else {
    const valueType = value.type; // Copied in here because of TS reasons, it's not available in the `default` case

    switch (valueType) {
      case 'boolean':
        return t.boolean;

      case 'keyword':
      case 'text':
      case 'date':
        return t.string;

      case 'byte':
      case 'double':
      case 'float':
      case 'integer':
      case 'long':
      case 'short':
        return t.number;

      case 'array':
        if ('items' in value) {
          return t.array(schemaValueToIoTs(value.items));
        }

        throw new Error(`Schema type must include the "items" declaration.`);

      default:
        throw new Error(`Unsupported schema type ${valueType}.`);
    }
  }
}
/**
 * Loops through a list of [key, SchemaValue] tuples to convert them into a valid io-ts parameter to define objects.
 * @param entries Array of tuples [key, {@link SchemaValue}]. Typically, coming from Object.entries(SchemaObject).
 */


function entriesToObjectIoTs(entries) {
  return Object.fromEntries(entries.map(([key, value]) => {
    try {
      return [key, schemaValueToIoTs(value)];
    } catch (err) {
      err.failedKey = [key, ...(err.failedKey || [])];
      throw err;
    }
  }));
}
/**
 * Converts a {@link SchemaObject} to the io-ts equivalent.
 * @param schemaObject The {@link SchemaObject} to parse.
 */


function schemaObjectToIoTs(schemaObject) {
  const objectEntries = Object.entries(schemaObject.properties);
  const requiredFields = objectEntries.filter(([key, {
    _meta
  }]) => (_meta === null || _meta === void 0 ? void 0 : _meta.optional) !== true);
  const optionalFields = objectEntries.filter(([key, {
    _meta
  }]) => (_meta === null || _meta === void 0 ? void 0 : _meta.optional) === true);
  return (0, _excess.excess)(t.intersection([t.interface(entriesToObjectIoTs(requiredFields)), t.partial(entriesToObjectIoTs(optionalFields))]));
}
/**
 * Converts a {@link RootSchema} to an io-ts validation object.
 * @param rootSchema The {@link RootSchema} to be parsed.
 */


function schemaToIoTs(rootSchema) {
  try {
    return schemaObjectToIoTs({
      properties: rootSchema
    });
  } catch (err) {
    if (err.failedKey) {
      err.message = `Malformed schema for key [${err.failedKey.join('.')}]: ${err.message}`;
    }

    throw err;
  }
}