/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { useEuiTheme } from '../../services/theme';
import { transparentize } from '../../services/color';
/**
 * Set scroll bar appearance on Chrome (and firefox).
 * All parameters are optional and default to specific global settings.
 */

export var euiScrollBarStyles = function euiScrollBarStyles(_ref) {
  var _ref$euiTheme = _ref.euiTheme,
      colors = _ref$euiTheme.colors,
      size = _ref$euiTheme.size;

  var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _thumbColor = _ref2.thumbColor,
      _ref2$trackColor = _ref2.trackColor,
      trackColor = _ref2$trackColor === void 0 ? 'transparent' : _ref2$trackColor,
      _ref2$width = _ref2.width,
      width = _ref2$width === void 0 ? 'thin' : _ref2$width,
      _size = _ref2.size,
      _corner = _ref2.corner;

  // Set defaults from theme
  var thumbColor = _thumbColor || transparentize(colors.darkShade, 0.5);

  var scrollBarSize = _size || size.base;
  var scrollBarCorner = _corner || width === 'thin' ? "calc(".concat(size.s, " * 0.75)") : size.xs; // Firefox's scrollbar coloring cascades, but the sizing does not,
  // so it's being added to this mixin for allowing support wherever custom scrollbars are

  var firefoxSupport = "scrollbar-color: ".concat(thumbColor, " ").concat(trackColor, ";");
  return "scrollbar-width: ".concat(width, ";\n\n    &::-webkit-scrollbar {\n      width: ").concat(scrollBarSize, ";\n      height: ").concat(scrollBarSize, ";\n    }\n\n    &::-webkit-scrollbar-thumb {\n      background-color: ").concat(thumbColor, ";\n      background-clip: content-box;\n      border-radius: ").concat(scrollBarSize, ";\n      border: ").concat(scrollBarCorner, " solid ").concat(trackColor, ";\n    }\n\n    &::-webkit-scrollbar-corner,\n    &::-webkit-scrollbar-track {\n      background-color: ").concat(trackColor, ";\n    }\n\n    ").concat(firefoxSupport, "\n  ");
};
export var useEuiScrollBar = function useEuiScrollBar(options) {
  var euiTheme = useEuiTheme();
  return euiScrollBarStyles(euiTheme, options);
};
/**
 * *INTERNAL*
 * Overflow shadow masks for use in YScroll and XScroll helpers
 */

var euiOverflowShadowStyles = function euiOverflowShadowStyles(_ref3) {
  var size = _ref3.euiTheme.size;

  var _ref4 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      _direction = _ref4.direction,
      _side = _ref4.side;

  var direction = _direction || 'y';
  var side = _side || 'both';
  var hideHeight = "calc(".concat(size.base, " * 0.75 * 1.25)");
  var gradientStart = "\n  ".concat(transparentize('red', 0.1), " 0%,\n  ").concat(transparentize('red', 1), " ").concat(hideHeight, "\n  ");
  var gradientEnd = "\n  ".concat(transparentize('red', 1), " calc(100% - ").concat(hideHeight, "),\n  ").concat(transparentize('red', 0.1), " 100%\n  ");
  var gradient = '';

  if (side) {
    if (side === 'both') {
      gradient = "".concat(gradientStart, ", ").concat(gradientEnd);
    } else if (side === 'start') {
      gradient = "".concat(gradientStart);
    } else {
      gradient = "".concat(gradientEnd);
    }
  }

  if (direction === 'y') {
    return "mask-image: linear-gradient(to bottom, ".concat(gradient, ");");
  } else {
    return "mask-image: linear-gradient(to right, ".concat(gradient, ");");
  }
};
/**
 * 1. Focus rings shouldn't be visible on scrollable regions, but a11y requires them to be focusable.
 *    Browser's supporting `:focus-visible` will still show outline on keyboard focus only.
 *    Others like Safari, won't show anything at all.
 */
// TODO: How do we use Emotion to output the CSS class utilities instead?


export var euiYScroll = function euiYScroll(euiTheme) {
  return "\n  ".concat(euiScrollBarStyles(euiTheme), "\n  height: 100%;\n  overflow-y: auto;\n  overflow-x: hidden;\n  &:focus {\n    outline: none; /* 1 */\n  }\n");
};
export var useEuiYScroll = function useEuiYScroll() {
  var euiTheme = useEuiTheme();
  return euiYScroll(euiTheme);
};
export var euiYScrollWithShadows = function euiYScrollWithShadows(euiTheme) {
  return "\n  ".concat(euiYScroll(euiTheme), "\n  ").concat(euiOverflowShadowStyles(euiTheme, {
    direction: 'y'
  }), "\n");
};
export var useEuiYScrollWithShadows = function useEuiYScrollWithShadows() {
  var euiTheme = useEuiTheme();
  return euiYScrollWithShadows(euiTheme);
};
export var euiXScroll = function euiXScroll(euiTheme) {
  return "\n  ".concat(euiScrollBarStyles(euiTheme), "\n  overflow-x: auto;\n  &:focus {\n    outline: none; /* 1 */\n  }\n");
};
export var useEuiXScroll = function useEuiXScroll() {
  var euiTheme = useEuiTheme();
  return euiXScroll(euiTheme);
};
export var euiXScrollWithShadows = function euiXScrollWithShadows(euiTheme) {
  return "\n  ".concat(euiXScroll(euiTheme), "\n  ").concat(euiOverflowShadowStyles(euiTheme, {
    direction: 'x'
  }), "\n");
};
export var useEuiXScrollWithShadows = function useEuiXScrollWithShadows() {
  var euiTheme = useEuiTheme();
  return euiXScrollWithShadows(euiTheme);
};
export var euiOverflowScroll = function euiOverflowScroll(euiTheme) {
  var _ref5 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
      direction = _ref5.direction,
      _ref5$mask = _ref5.mask,
      mask = _ref5$mask === void 0 ? false : _ref5$mask;

  switch (direction) {
    case 'y':
      return mask ? euiYScrollWithShadows(euiTheme) : euiYScroll(euiTheme);

    case 'x':
      return mask ? euiXScrollWithShadows(euiTheme) : euiXScroll(euiTheme);

    default:
      console.warn('Please provide a valid direction option to useEuiOverflowScroll');
      return '';
  }
};
export var useEuiOverflowScroll = function useEuiOverflowScroll(direction) {
  var mask = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
  var euiTheme = useEuiTheme();
  return euiOverflowScroll(euiTheme, {
    direction: direction,
    mask: mask
  });
};