"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getActionList = void 0;

var _constants = require("../../../../common/endpoint/constants");

var _custom_http_request_error = require("../../../utils/custom_http_request_error");

var _action_list_helpers = require("../../utils/action_list_helpers");

var _utils = require("./utils");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const getActionList = async ({
  commands,
  elasticAgentIds,
  esClient,
  endDate,
  logger,
  page: _page,
  pageSize,
  startDate,
  userIds
}) => {
  const size = pageSize !== null && pageSize !== void 0 ? pageSize : _constants.ENDPOINT_DEFAULT_PAGE_SIZE;
  const page = _page !== null && _page !== void 0 ? _page : 1; // # of hits to skip

  const from = (page - 1) * size;
  const {
    actionDetails,
    totalRecords
  } = await getActionDetailsList({
    commands,
    elasticAgentIds,
    esClient,
    endDate,
    from,
    logger,
    size,
    startDate,
    userIds
  });
  return {
    page,
    pageSize: size,
    startDate,
    endDate,
    elasticAgentIds,
    userIds,
    commands,
    data: actionDetails,
    total: totalRecords
  };
};

exports.getActionList = getActionList;

const getActionDetailsList = async ({
  commands,
  elasticAgentIds,
  esClient,
  endDate,
  from,
  logger,
  size,
  startDate,
  userIds
}) => {
  var _actionRequests2, _actionRequests2$body, _actionRequests2$body2, _actionRequests3, _actionRequests3$body, _actionRequests3$body2, _actionRequests4, _actionRequests4$body, _actionRequests4$body2, _actionResponses, _actionResponses$body, _actionResponses$body2;

  let actionRequests;
  let actionReqIds;
  let actionResponses;

  try {
    // fetch actions with matching agent_ids if any
    const {
      actionIds,
      actionRequests: _actionRequests
    } = await (0, _action_list_helpers.getActions)({
      commands,
      esClient,
      elasticAgentIds,
      startDate,
      endDate,
      from,
      size,
      userIds
    });
    actionRequests = _actionRequests;
    actionReqIds = actionIds;
  } catch (error) {
    var _error$meta$meta$body, _error$meta, _error$meta$meta, _error$meta$meta$body2, _error$meta$meta$body3, _error$meta$meta$stat, _error$meta2, _error$meta2$meta;

    // all other errors
    const err = new _custom_http_request_error.CustomHttpRequestError((_error$meta$meta$body = (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$meta = _error$meta.meta) === null || _error$meta$meta === void 0 ? void 0 : (_error$meta$meta$body2 = _error$meta$meta.body) === null || _error$meta$meta$body2 === void 0 ? void 0 : (_error$meta$meta$body3 = _error$meta$meta$body2.error) === null || _error$meta$meta$body3 === void 0 ? void 0 : _error$meta$meta$body3.reason) !== null && _error$meta$meta$body !== void 0 ? _error$meta$meta$body : 'Unknown error while fetching action requests', (_error$meta$meta$stat = (_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$meta = _error$meta2.meta) === null || _error$meta2$meta === void 0 ? void 0 : _error$meta2$meta.statusCode) !== null && _error$meta$meta$stat !== void 0 ? _error$meta$meta$stat : 500, error);
    logger.error(err);
    throw err;
  } // return empty details array


  if (!((_actionRequests2 = actionRequests) !== null && _actionRequests2 !== void 0 && (_actionRequests2$body = _actionRequests2.body) !== null && _actionRequests2$body !== void 0 && (_actionRequests2$body2 = _actionRequests2$body.hits) !== null && _actionRequests2$body2 !== void 0 && _actionRequests2$body2.hits)) return {
    actionDetails: [],
    totalRecords: 0
  }; // format endpoint actions into { type, item } structure

  const formattedActionRequests = (0, _utils.formatEndpointActionResults)((_actionRequests3 = actionRequests) === null || _actionRequests3 === void 0 ? void 0 : (_actionRequests3$body = _actionRequests3.body) === null || _actionRequests3$body === void 0 ? void 0 : (_actionRequests3$body2 = _actionRequests3$body.hits) === null || _actionRequests3$body2 === void 0 ? void 0 : _actionRequests3$body2.hits);
  const totalRecords = ((_actionRequests4 = actionRequests) === null || _actionRequests4 === void 0 ? void 0 : (_actionRequests4$body = _actionRequests4.body) === null || _actionRequests4$body === void 0 ? void 0 : (_actionRequests4$body2 = _actionRequests4$body.hits) === null || _actionRequests4$body2 === void 0 ? void 0 : _actionRequests4$body2.total).value; // normalized actions with a flat structure to access relevant values

  const normalizedActionRequests = formattedActionRequests.map(action => (0, _utils.mapToNormalizedActionRequest)(action.item.data));

  try {
    // get all responses for given action Ids and agent Ids
    actionResponses = await (0, _action_list_helpers.getActionResponses)({
      actionIds: actionReqIds,
      elasticAgentIds,
      esClient
    });
  } catch (error) {
    var _error$meta$meta$body4, _error$meta3, _error$meta3$meta, _error$meta3$meta$bod, _error$meta3$meta$bod2, _error$meta$meta$stat2, _error$meta4, _error$meta4$meta;

    // all other errors
    const err = new _custom_http_request_error.CustomHttpRequestError((_error$meta$meta$body4 = (_error$meta3 = error.meta) === null || _error$meta3 === void 0 ? void 0 : (_error$meta3$meta = _error$meta3.meta) === null || _error$meta3$meta === void 0 ? void 0 : (_error$meta3$meta$bod = _error$meta3$meta.body) === null || _error$meta3$meta$bod === void 0 ? void 0 : (_error$meta3$meta$bod2 = _error$meta3$meta$bod.error) === null || _error$meta3$meta$bod2 === void 0 ? void 0 : _error$meta3$meta$bod2.reason) !== null && _error$meta$meta$body4 !== void 0 ? _error$meta$meta$body4 : 'Unknown error while fetching action responses', (_error$meta$meta$stat2 = (_error$meta4 = error.meta) === null || _error$meta4 === void 0 ? void 0 : (_error$meta4$meta = _error$meta4.meta) === null || _error$meta4$meta === void 0 ? void 0 : _error$meta4$meta.statusCode) !== null && _error$meta$meta$stat2 !== void 0 ? _error$meta$meta$stat2 : 500, error);
    logger.error(err);
    throw err;
  } // categorize responses as fleet and endpoint responses


  const categorizedResponses = (0, _utils.categorizeResponseResults)({
    results: (_actionResponses = actionResponses) === null || _actionResponses === void 0 ? void 0 : (_actionResponses$body = _actionResponses.body) === null || _actionResponses$body === void 0 ? void 0 : (_actionResponses$body2 = _actionResponses$body.hits) === null || _actionResponses$body2 === void 0 ? void 0 : _actionResponses$body2.hits
  }); // compute action details list for each action id

  const actionDetails = normalizedActionRequests.map(action => {
    // pick only those responses that match the current action id
    const matchedResponses = categorizedResponses.filter(categorizedResponse => categorizedResponse.type === 'response' ? categorizedResponse.item.data.EndpointActions.action_id === action.id : categorizedResponse.item.data.action_id === action.id); // find the specific response's details using that set of matching responses

    const {
      isCompleted,
      completedAt,
      wasSuccessful,
      errors
    } = (0, _utils.getActionCompletionInfo)(action.agents, matchedResponses);
    return {
      id: action.id,
      agents: action.agents,
      command: action.command,
      startedAt: action.createdAt,
      isCompleted,
      completedAt,
      wasSuccessful,
      errors,
      isExpired: !isCompleted && action.expiration < new Date().toISOString(),
      createdBy: action.createdBy,
      comment: action.comment,
      parameters: action.parameters
    };
  });
  return {
    actionDetails,
    totalRecords
  };
};