"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sendEndpointMetadataUpdate = void 0;

var _lodash = require("lodash");

var _constants = require("../../../common/endpoint/constants");

var _generate_data = require("../../../common/endpoint/generate_data");

var _fleet_services = require("./fleet_services");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const endpointGenerator = new _generate_data.EndpointDocGenerator();

const sendEndpointMetadataUpdate = async (esClient, agentId, overrides = {}, {
  checkInAgent = true
} = {}) => {
  const lastStreamedDoc = await fetchLastStreamedEndpointUpdate(esClient, agentId);

  if (!lastStreamedDoc) {
    throw new Error(`An endpoint with agent.id of [${agentId}] not found!`);
  }

  if (checkInAgent) {
    // Trigger an agent checkin and just let it run
    (0, _fleet_services.checkInFleetAgent)(esClient, agentId);
  }

  const generatedHostMetadataDoc = (0, _lodash.clone)(endpointGenerator.generateHostMetadata());
  const newUpdate = (0, _lodash.merge)(lastStreamedDoc, {
    event: generatedHostMetadataDoc.event,
    // Make sure to use a new event object
    '@timestamp': generatedHostMetadataDoc['@timestamp']
  }, overrides);
  return esClient.index({
    index: _constants.METADATA_DATASTREAM,
    body: newUpdate,
    op_type: 'create'
  });
};

exports.sendEndpointMetadataUpdate = sendEndpointMetadataUpdate;

const fetchLastStreamedEndpointUpdate = async (esClient, agentId) => {
  var _queryResult$hits, _queryResult$hits$hit;

  const queryResult = await esClient.search({
    index: _constants.METADATA_DATASTREAM,
    size: 1,
    body: {
      query: {
        bool: {
          filter: [{
            bool: {
              should: [{
                term: {
                  'elastic.agent.id': agentId
                }
              }]
            }
          }]
        }
      },
      // Am I doing this right? I want only the last document for the host.id that was sent
      collapse: {
        field: 'host.id',
        inner_hits: {
          name: 'most_recent',
          size: 1,
          sort: [{
            'event.created': 'desc'
          }]
        }
      },
      aggs: {
        total: {
          cardinality: {
            field: 'host.id'
          }
        }
      },
      sort: [{
        'event.created': {
          order: 'desc'
        }
      }]
    }
  }, {
    ignore: [404]
  });
  return (_queryResult$hits = queryResult.hits) === null || _queryResult$hits === void 0 ? void 0 : (_queryResult$hits$hit = _queryResult$hits.hits[0]) === null || _queryResult$hits$hit === void 0 ? void 0 : _queryResult$hits$hit._source;
};