"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerConnectorRoutes = registerConnectorRoutes;

var _configSchema = require("@kbn/config-schema");

var _i18n = require("@kbn/i18n");

var _error_codes = require("../../../common/types/error_codes");

var _add_connector = require("../../lib/connectors/add_connector");

var _start_sync = require("../../lib/connectors/start_sync");

var _update_connector_configuration = require("../../lib/connectors/update_connector_configuration");

var _update_connector_scheduling = require("../../lib/connectors/update_connector_scheduling");

var _create_error = require("../../utils/create_error");

var _elasticsearch_error_handler = require("../../utils/elasticsearch_error_handler");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
function registerConnectorRoutes({
  router,
  log
}) {
  router.post({
    path: '/internal/enterprise_search/connectors',
    validate: {
      body: _configSchema.schema.object({
        delete_existing_connector: _configSchema.schema.maybe(_configSchema.schema.boolean()),
        index_name: _configSchema.schema.string(),
        language: _configSchema.schema.nullable(_configSchema.schema.string())
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;

    try {
      const body = await (0, _add_connector.addConnector)(client, request.body);
      return response.ok({
        body
      });
    } catch (error) {
      if (error.message === _error_codes.ErrorCode.CONNECTOR_DOCUMENT_ALREADY_EXISTS || error.message === _error_codes.ErrorCode.INDEX_ALREADY_EXISTS) {
        return (0, _create_error.createError)({
          errorCode: error.message,
          message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.addConnector.connectorExistsError', {
            defaultMessage: 'Connector or index already exists'
          }),
          response,
          statusCode: 409
        });
      }

      throw error;
    }
  }));
  router.post({
    path: '/internal/enterprise_search/connectors/{connectorId}/configuration',
    validate: {
      body: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.object({
        label: _configSchema.schema.string(),
        value: _configSchema.schema.string()
      })),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    await (0, _update_connector_configuration.updateConnectorConfiguration)(client, request.params.connectorId, request.body);
    return response.ok();
  }));
  router.post({
    path: '/internal/enterprise_search/connectors/{connectorId}/scheduling',
    validate: {
      body: _configSchema.schema.object({
        enabled: _configSchema.schema.boolean(),
        interval: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    await (0, _update_connector_scheduling.updateConnectorScheduling)(client, request.params.connectorId, request.body);
    return response.ok();
  }));
  router.post({
    path: '/internal/enterprise_search/connectors/{connectorId}/start_sync',
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    await (0, _start_sync.startConnectorSync)(client, request.params.connectorId);
    return response.ok();
  }));
}