"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addConnector = void 0;

var _ = require("../..");

var _connectors = require("../../../common/types/connectors");

var _error_codes = require("../../../common/types/error_codes");

var _setup_indices = require("../../index_management/setup_indices");

var _fetch_crawlers = require("../crawler/fetch_crawlers");

var _text_analysis = require("../indices/text_analysis");

var _delete_connector = require("./delete_connector");

var _fetch_connectors = require("./fetch_connectors");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const createConnector = async (document, client, language, deleteExisting) => {
  const index = document.index_name;
  const indexExists = await client.asCurrentUser.indices.exists({
    index
  });

  if (indexExists) {
    {
      throw new Error(_error_codes.ErrorCode.INDEX_ALREADY_EXISTS);
    }
  }

  const connector = await (0, _fetch_connectors.fetchConnectorByIndexName)(client, index);

  if (connector) {
    if (deleteExisting) {
      await (0, _delete_connector.deleteConnectorById)(client, connector.id);
    } else {
      throw new Error(_error_codes.ErrorCode.CONNECTOR_DOCUMENT_ALREADY_EXISTS);
    }
  }

  const crawler = await (0, _fetch_crawlers.fetchCrawlerByIndexName)(client, index);

  if (crawler) {
    throw new Error(_error_codes.ErrorCode.CRAWLER_ALREADY_EXISTS);
  }

  const result = await client.asCurrentUser.index({
    document,
    index: _.CONNECTORS_INDEX
  });
  await client.asCurrentUser.indices.create({
    index,
    settings: (0, _text_analysis.textAnalysisSettings)(language !== null && language !== void 0 ? language : undefined)
  });
  await client.asCurrentUser.indices.refresh({
    index: _.CONNECTORS_INDEX
  });
  return {
    id: result._id,
    index_name: document.index_name
  };
};

const addConnector = async (client, input) => {
  const document = {
    api_key_id: null,
    configuration: {},
    index_name: input.index_name,
    language: input.language,
    last_seen: null,
    last_sync_error: null,
    last_sync_status: null,
    last_synced: null,
    name: input.index_name.startsWith('search-') ? input.index_name.substring(7) : input.index_name,
    scheduling: {
      enabled: false,
      interval: '0 0 0 * * ?'
    },
    service_type: null,
    status: _connectors.ConnectorStatus.CREATED,
    sync_now: false
  };
  const connectorsIndexExists = await client.asCurrentUser.indices.exists({
    index: _.CONNECTORS_INDEX
  });

  if (!connectorsIndexExists) {
    await (0, _setup_indices.setupConnectorsIndices)(client.asCurrentUser);
  }

  return await createConnector(document, client, input.language, !!input.delete_existing_connector);
};

exports.addConnector = addConnector;