"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTopDependencyOperations = getTopDependencyOperations;

var _server = require("../../../../observability/server");

var _elasticsearch_fieldnames = require("../../../common/elasticsearch_fieldnames");

var _event_outcome = require("../../../common/event_outcome");

var _processor_event = require("../../../common/processor_event");

var _environment_query = require("../../../common/utils/environment_query");

var _get_offset_in_ms = require("../../../common/utils/get_offset_in_ms");

var _calculate_throughput = require("../../lib/helpers/calculate_throughput");

var _metrics = require("../../lib/helpers/metrics");

var _calculate_impact_builder = require("../traces/calculate_impact_builder");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const MAX_NUM_OPERATIONS = 500;

async function getTopDependencyOperations({
  setup,
  dependencyName,
  start,
  end,
  offset,
  environment,
  kuery
}) {
  var _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4;

  const {
    apmEventClient
  } = setup;
  const {
    startWithOffset,
    endWithOffset,
    offsetInMs
  } = (0, _get_offset_in_ms.getOffsetInMs)({
    start,
    end,
    offset
  });
  const aggs = {
    duration: {
      avg: {
        field: _elasticsearch_fieldnames.SPAN_DURATION
      }
    },
    successful: {
      filter: {
        term: {
          [_elasticsearch_fieldnames.EVENT_OUTCOME]: _event_outcome.EventOutcome.success
        }
      }
    },
    failure: {
      filter: {
        term: {
          [_elasticsearch_fieldnames.EVENT_OUTCOME]: _event_outcome.EventOutcome.failure
        }
      }
    }
  };
  const response = await apmEventClient.search('get_top_dependency_operations', {
    apm: {
      events: [_processor_event.ProcessorEvent.span]
    },
    body: {
      size: 0,
      query: {
        bool: {
          filter: [...(0, _server.rangeQuery)(startWithOffset, endWithOffset), ...(0, _environment_query.environmentQuery)(environment), ...(0, _server.kqlQuery)(kuery), ...(0, _server.termQuery)(_elasticsearch_fieldnames.SPAN_DESTINATION_SERVICE_RESOURCE, dependencyName)]
        }
      },
      aggs: {
        operationName: {
          terms: {
            field: _elasticsearch_fieldnames.SPAN_NAME,
            size: MAX_NUM_OPERATIONS
          },
          aggs: {
            over_time: {
              date_histogram: (0, _metrics.getMetricsDateHistogramParams)({
                start: startWithOffset,
                end: endWithOffset,
                metricsInterval: 60
              }),
              aggs
            },
            ...aggs,
            total_time: {
              sum: {
                field: _elasticsearch_fieldnames.SPAN_DURATION
              }
            }
          }
        }
      }
    }
  });
  const getImpact = (0, _calculate_impact_builder.calculateImpactBuilder)((_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.operationName.buckets.map(bucket => bucket.total_time.value)) !== null && _response$aggregation !== void 0 ? _response$aggregation : []);
  return (_response$aggregation3 = (_response$aggregation4 = response.aggregations) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.operationName.buckets.map(bucket => {
    var _bucket$total_time$va;

    const timeseries = {
      latency: [],
      throughput: [],
      failureRate: []
    };
    bucket.over_time.buckets.forEach(dateBucket => {
      const x = dateBucket.key + offsetInMs;
      timeseries.throughput.push({
        x,
        y: (0, _calculate_throughput.calculateThroughputWithInterval)({
          value: dateBucket.doc_count,
          bucketSize: 60
        })
      });
      timeseries.latency.push({
        x,
        y: dateBucket.duration.value
      });
      timeseries.failureRate.push({
        x,
        y: dateBucket.failure.doc_count > 0 || dateBucket.successful.doc_count > 0 ? dateBucket.failure.doc_count / (dateBucket.successful.doc_count + dateBucket.failure.doc_count) : null
      });
    });
    return {
      spanName: bucket.key,
      latency: bucket.duration.value,
      throughput: (0, _calculate_throughput.calculateThroughputWithRange)({
        start: startWithOffset,
        end: endWithOffset,
        value: bucket.doc_count
      }),
      failureRate: bucket.failure.doc_count > 0 || bucket.successful.doc_count > 0 ? bucket.failure.doc_count / (bucket.successful.doc_count + bucket.failure.doc_count) : null,
      impact: getImpact((_bucket$total_time$va = bucket.total_time.value) !== null && _bucket$total_time$va !== void 0 ? _bucket$total_time$va : 0),
      timeseries
    };
  })) !== null && _response$aggregation3 !== void 0 ? _response$aggregation3 : [];
}