"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineExplainLogRateSpikesRoute = void 0;

var _lodash = require("lodash");

var _i18n = require("@kbn/i18n");

var _std = require("@kbn/std");

var _fieldTypes = require("@kbn/field-types");

var _aiopsUtils = require("@kbn/aiops-utils");

var _mlAggUtils = require("@kbn/ml-agg-utils");

var _explain_log_rate_spikes = require("../../common/api/explain_log_rate_spikes");

var _api = require("../../common/api");

var _fetch_field_candidates = require("./queries/fetch_field_candidates");

var _fetch_change_point_p_values = require("./queries/fetch_change_point_p_values");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
// Overall progress is a float from 0 to 1.
const LOADED_FIELD_CANDIDATES = 0.2;
const PROGRESS_STEP_P_VALUES = 0.6;
const PROGRESS_STEP_HISTOGRAMS = 0.2;

const defineExplainLogRateSpikesRoute = (router, license, logger) => {
  router.post({
    path: _api.API_ENDPOINT.EXPLAIN_LOG_RATE_SPIKES,
    validate: {
      body: _explain_log_rate_spikes.aiopsExplainLogRateSpikesSchema
    }
  }, async (context, request, response) => {
    if (!license.isActivePlatinumLicense) {
      return response.forbidden();
    }

    const client = (await context.core).elasticsearch.client.asCurrentUser;
    const controller = new AbortController();
    let loaded = 0;
    let shouldStop = false;
    request.events.aborted$.subscribe(() => {
      shouldStop = true;
      controller.abort();
    });
    request.events.completed$.subscribe(() => {
      shouldStop = true;
      controller.abort();
    });
    const {
      end,
      push,
      responseWithHeaders
    } = (0, _aiopsUtils.streamFactory)(request.headers, logger);

    function endWithUpdatedLoadingState() {
      push((0, _explain_log_rate_spikes.updateLoadingStateAction)({
        ccsWarning: false,
        loaded: 1,
        loadingState: _i18n.i18n.translate('xpack.aiops.explainLogRateSpikes.loadingState.doneMessage', {
          defaultMessage: 'Done.'
        })
      }));
      end();
    } // Async IIFE to run the analysis while not blocking returning `responseWithHeaders`.


    (async () => {
      push((0, _explain_log_rate_spikes.resetAction)());
      push((0, _explain_log_rate_spikes.updateLoadingStateAction)({
        ccsWarning: false,
        loaded,
        loadingState: _i18n.i18n.translate('xpack.aiops.explainLogRateSpikes.loadingState.loadingFieldCandidates', {
          defaultMessage: 'Loading field candidates.'
        })
      }));
      let fieldCandidates;

      try {
        fieldCandidates = await (0, _fetch_field_candidates.fetchFieldCandidates)(client, request.body);
      } catch (e) {
        push((0, _explain_log_rate_spikes.addErrorAction)(e.toString()));
        end();
        return;
      }

      loaded += LOADED_FIELD_CANDIDATES;
      push((0, _explain_log_rate_spikes.updateLoadingStateAction)({
        ccsWarning: false,
        loaded,
        loadingState: _i18n.i18n.translate('xpack.aiops.explainLogRateSpikes.loadingState.identifiedFieldCandidates', {
          defaultMessage: 'Identified {fieldCandidatesCount, plural, one {# field candidate} other {# field candidates}}.',
          values: {
            fieldCandidatesCount: fieldCandidates.length
          }
        })
      }));

      if (fieldCandidates.length === 0) {
        endWithUpdatedLoadingState();
      } else if (shouldStop) {
        end();
        return;
      }

      const changePoints = [];
      const fieldsToSample = new Set();
      const chunkSize = 10;
      const fieldCandidatesChunks = (0, _lodash.chunk)(fieldCandidates, chunkSize);

      for (const fieldCandidatesChunk of fieldCandidatesChunks) {
        var _changePoints$length;

        let pValues;

        try {
          pValues = await (0, _fetch_change_point_p_values.fetchChangePointPValues)(client, request.body, fieldCandidatesChunk);
        } catch (e) {
          push((0, _explain_log_rate_spikes.addErrorAction)(e.toString()));
          end();
          return;
        }

        if (pValues.length > 0) {
          pValues.forEach(d => {
            fieldsToSample.add(d.fieldName);
          });
          changePoints.push(...pValues);
        }

        loaded += 1 / fieldCandidatesChunks.length * PROGRESS_STEP_P_VALUES;

        if (pValues.length > 0) {
          push((0, _explain_log_rate_spikes.addChangePointsAction)(pValues));
        }

        push((0, _explain_log_rate_spikes.updateLoadingStateAction)({
          ccsWarning: false,
          loaded,
          loadingState: _i18n.i18n.translate('xpack.aiops.explainLogRateSpikes.loadingState.identifiedFieldValuePairs', {
            defaultMessage: 'Identified {fieldValuePairsCount, plural, one {# significant field/value pair} other {# significant field/value pairs}}.',
            values: {
              fieldValuePairsCount: (_changePoints$length = changePoints === null || changePoints === void 0 ? void 0 : changePoints.length) !== null && _changePoints$length !== void 0 ? _changePoints$length : 0
            }
          })
        }));

        if (shouldStop) {
          end();
          return;
        }
      }

      if ((changePoints === null || changePoints === void 0 ? void 0 : changePoints.length) === 0) {
        endWithUpdatedLoadingState();
        return;
      }

      const histogramFields = [{
        fieldName: request.body.timeFieldName,
        type: _fieldTypes.KBN_FIELD_TYPES.DATE
      }];
      const [overallTimeSeries] = await (0, _mlAggUtils.fetchHistogramsForFields)(client, request.body.index, {
        match_all: {}
      }, // fields
      histogramFields, // samplerShardSize
      -1, undefined); // time series filtered by fields

      if (changePoints) {
        await (0, _std.asyncForEach)(changePoints, async (cp, index) => {
          if (changePoints) {
            var _overallTimeSeries$da;

            const histogramQuery = {
              bool: {
                filter: [{
                  term: {
                    [cp.fieldName]: cp.fieldValue
                  }
                }]
              }
            };
            const [cpTimeSeries] = await (0, _mlAggUtils.fetchHistogramsForFields)(client, request.body.index, histogramQuery, // fields
            [{
              fieldName: request.body.timeFieldName,
              type: _fieldTypes.KBN_FIELD_TYPES.DATE,
              interval: overallTimeSeries.interval,
              min: overallTimeSeries.stats[0],
              max: overallTimeSeries.stats[1]
            }], // samplerShardSize
            -1, undefined);
            const histogram = (_overallTimeSeries$da = overallTimeSeries.data.map((o, i) => {
              var _cpTimeSeries$data$fi, _o$key_as_string;

              const current = (_cpTimeSeries$data$fi = cpTimeSeries.data.find(d1 => d1.key_as_string === o.key_as_string)) !== null && _cpTimeSeries$data$fi !== void 0 ? _cpTimeSeries$data$fi : {
                doc_count: 0
              };
              return {
                key: o.key,
                key_as_string: (_o$key_as_string = o.key_as_string) !== null && _o$key_as_string !== void 0 ? _o$key_as_string : '',
                doc_count_change_point: current.doc_count,
                doc_count_overall: Math.max(0, o.doc_count - current.doc_count)
              };
            })) !== null && _overallTimeSeries$da !== void 0 ? _overallTimeSeries$da : [];
            const {
              fieldName,
              fieldValue
            } = cp;
            loaded += 1 / changePoints.length * PROGRESS_STEP_HISTOGRAMS;
            push((0, _explain_log_rate_spikes.updateLoadingStateAction)({
              ccsWarning: false,
              loaded,
              loadingState: _i18n.i18n.translate('xpack.aiops.explainLogRateSpikes.loadingState.loadingHistogramData', {
                defaultMessage: 'Loading histogram data.'
              })
            }));
            push((0, _explain_log_rate_spikes.addChangePointsHistogramAction)([{
              fieldName,
              fieldValue,
              histogram
            }]));
          }
        });
      }

      endWithUpdatedLoadingState();
    })();

    return response.ok(responseWithHeaders);
  });
};

exports.defineExplainLogRateSpikesRoute = defineExplainLogRateSpikesRoute;