"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.toMetric = exports.toScopeMetrics = exports.toResourceMetrics = void 0;
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const api_metrics_1 = require("@opentelemetry/api-metrics");
const core_1 = require("@opentelemetry/core");
const sdk_metrics_base_1 = require("@opentelemetry/sdk-metrics-base");
const internal_1 = require("../common/internal");
function toResourceMetrics(resourceMetrics) {
    return {
        resource: {
            attributes: (0, internal_1.toAttributes)(resourceMetrics.resource.attributes),
            droppedAttributesCount: 0
        },
        schemaUrl: undefined,
        scopeMetrics: toScopeMetrics(resourceMetrics.scopeMetrics)
    };
}
exports.toResourceMetrics = toResourceMetrics;
function toScopeMetrics(scopeMetrics) {
    return Array.from(scopeMetrics.map(metrics => {
        const scopeMetrics = {
            scope: {
                name: metrics.scope.name,
                version: metrics.scope.version,
            },
            metrics: metrics.metrics.map(metricData => toMetric(metricData)),
            schemaUrl: metrics.scope.schemaUrl
        };
        return scopeMetrics;
    }));
}
exports.toScopeMetrics = toScopeMetrics;
function toMetric(metricData) {
    const out = {
        name: metricData.descriptor.name,
        description: metricData.descriptor.description,
        unit: metricData.descriptor.unit,
    };
    const aggregationTemporality = toAggregationTemporality(metricData.aggregationTemporality);
    if (metricData.dataPointType === sdk_metrics_base_1.DataPointType.SINGULAR) {
        const dataPoints = toSingularDataPoints(metricData);
        const isMonotonic = metricData.descriptor.type === sdk_metrics_base_1.InstrumentType.COUNTER ||
            metricData.descriptor.type === sdk_metrics_base_1.InstrumentType.OBSERVABLE_COUNTER;
        if (isSum(metricData)) {
            out.sum = {
                aggregationTemporality,
                isMonotonic,
                dataPoints
            };
        }
        else {
            // Instrument is a gauge.
            out.gauge = {
                dataPoints
            };
        }
    }
    else if (isHistogram(metricData)) {
        out.histogram = {
            aggregationTemporality,
            dataPoints: toHistogramDataPoints(metricData)
        };
    }
    return out;
}
exports.toMetric = toMetric;
function toSingularDataPoint(dataPoint, valueType) {
    const out = {
        attributes: (0, internal_1.toAttributes)(dataPoint.attributes),
        startTimeUnixNano: (0, core_1.hrTimeToNanoseconds)(dataPoint.startTime),
        timeUnixNano: (0, core_1.hrTimeToNanoseconds)(dataPoint.endTime),
    };
    if (valueType === api_metrics_1.ValueType.INT) {
        out.asInt = dataPoint.value;
    }
    else if (valueType === api_metrics_1.ValueType.DOUBLE) {
        out.asDouble = dataPoint.value;
    }
    return out;
}
function toSingularDataPoints(metricData) {
    return metricData.dataPoints.map(dataPoint => {
        return toSingularDataPoint(dataPoint, metricData.descriptor.valueType);
    });
}
function toHistogramDataPoints(metricData) {
    return metricData.dataPoints.map(dataPoint => {
        const histogram = dataPoint.value;
        return {
            attributes: (0, internal_1.toAttributes)(dataPoint.attributes),
            bucketCounts: histogram.buckets.counts,
            explicitBounds: histogram.buckets.boundaries,
            count: histogram.count,
            sum: histogram.sum,
            min: histogram.hasMinMax ? histogram.min : undefined,
            max: histogram.hasMinMax ? histogram.max : undefined,
            startTimeUnixNano: (0, core_1.hrTimeToNanoseconds)(dataPoint.startTime),
            timeUnixNano: (0, core_1.hrTimeToNanoseconds)(dataPoint.endTime),
        };
    });
}
function isSum(metric) {
    return (metric.descriptor.type === sdk_metrics_base_1.InstrumentType.COUNTER ||
        metric.descriptor.type === sdk_metrics_base_1.InstrumentType.UP_DOWN_COUNTER ||
        metric.descriptor.type === sdk_metrics_base_1.InstrumentType.OBSERVABLE_COUNTER ||
        metric.descriptor.type === sdk_metrics_base_1.InstrumentType.OBSERVABLE_UP_DOWN_COUNTER);
}
function isHistogram(metric) {
    return metric.dataPointType === sdk_metrics_base_1.DataPointType.HISTOGRAM;
}
function toAggregationTemporality(temporality) {
    if (temporality === sdk_metrics_base_1.AggregationTemporality.DELTA) {
        return 1 /* AGGREGATION_TEMPORALITY_DELTA */;
    }
    if (temporality === sdk_metrics_base_1.AggregationTemporality.CUMULATIVE) {
        return 2 /* AGGREGATION_TEMPORALITY_CUMULATIVE */;
    }
    return 0 /* AGGREGATION_TEMPORALITY_UNSPECIFIED */;
}
//# sourceMappingURL=internal.js.map