import _objectWithoutProperties from "@babel/runtime/helpers/objectWithoutProperties";
import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
import _objectSpread from "@babel/runtime/helpers/objectSpread2";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
var _excluded = ["meta"];

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import uuid from 'uuid';
import { addIdToItem, removeIdFromItem } from '@kbn/securitysolution-utils';
import { validate } from '@kbn/securitysolution-io-ts-utils';
import { ListOperatorEnum as OperatorEnum, ListOperatorTypeEnum as OperatorTypeEnum, createExceptionListItemSchema, entriesList, entriesNested, entry, exceptionListItemSchema, nestedEntryItem } from '@kbn/securitysolution-io-ts-list-types';
import { getDataViewFieldSubtypeNested, isDataViewFieldSubtypeNested } from '@kbn/es-query';
import { ALL_OPERATORS, EXCEPTION_OPERATORS_SANS_LISTS, doesNotExistOperator, existsOperator, isNotOperator, isOneOfOperator, isOperator, DETECTION_ENGINE_EXCEPTION_OPERATORS, isNotOneOfOperator, isInListOperator, isNotInListOperator } from '../autocomplete_operators';
export var isEntryNested = function isEntryNested(item) {
  return item.entries != null;
};
export var filterExceptionItems = function filterExceptionItems(exceptions) {
  return exceptions.reduce(function (acc, exception) {
    var entries = exception.entries.reduce(function (nestedAcc, singleEntry) {
      var strippedSingleEntry = removeIdFromItem(singleEntry);

      if (entriesNested.is(strippedSingleEntry)) {
        var nestedEntriesArray = strippedSingleEntry.entries.filter(function (singleNestedEntry) {
          var noIdSingleNestedEntry = removeIdFromItem(singleNestedEntry);

          var _validate = validate(noIdSingleNestedEntry, nestedEntryItem),
              _validate2 = _slicedToArray(_validate, 1),
              validatedNestedEntry = _validate2[0];

          return validatedNestedEntry != null;
        });
        var noIdNestedEntries = nestedEntriesArray.map(function (singleNestedEntry) {
          return removeIdFromItem(singleNestedEntry);
        });

        var _validate3 = validate(_objectSpread(_objectSpread({}, strippedSingleEntry), {}, {
          entries: noIdNestedEntries
        }), entriesNested),
            _validate4 = _slicedToArray(_validate3, 1),
            validatedNestedEntry = _validate4[0];

        if (validatedNestedEntry != null) {
          return [].concat(_toConsumableArray(nestedAcc), [_objectSpread(_objectSpread({}, singleEntry), {}, {
            entries: nestedEntriesArray
          })]);
        }

        return nestedAcc;
      } else {
        var _validate5 = validate(strippedSingleEntry, entry),
            _validate6 = _slicedToArray(_validate5, 1),
            validatedEntry = _validate6[0];

        if (validatedEntry != null) {
          return [].concat(_toConsumableArray(nestedAcc), [singleEntry]);
        }

        return nestedAcc;
      }
    }, []);

    if (entries.length === 0) {
      return acc;
    }

    var item = _objectSpread(_objectSpread({}, exception), {}, {
      entries: entries
    });

    if (exceptionListItemSchema.is(item)) {
      return [].concat(_toConsumableArray(acc), [item]);
    } else if (createExceptionListItemSchema.is(item)) {
      var meta = item.meta,
          rest = _objectWithoutProperties(item, _excluded);

      var itemSansMetaId = _objectSpread(_objectSpread({}, rest), {}, {
        meta: undefined
      });

      return [].concat(_toConsumableArray(acc), [itemSansMetaId]);
    } else {
      return acc;
    }
  }, []);
};
export var addIdToEntries = function addIdToEntries(entries) {
  return entries.map(function (singleEntry) {
    if (singleEntry.type === 'nested') {
      return addIdToItem(_objectSpread(_objectSpread({}, singleEntry), {}, {
        entries: singleEntry.entries.map(function (nestedEntry) {
          return addIdToItem(nestedEntry);
        })
      }));
    } else {
      return addIdToItem(singleEntry);
    }
  });
};
export var getNewExceptionItem = function getNewExceptionItem(_ref) {
  var listId = _ref.listId,
      namespaceType = _ref.namespaceType,
      ruleName = _ref.ruleName;
  return {
    comments: [],
    description: "".concat(ruleName, " - exception list item"),
    entries: addIdToEntries([{
      field: '',
      operator: 'included',
      type: 'match',
      value: ''
    }]),
    item_id: undefined,
    list_id: listId,
    meta: {
      temporaryUuid: uuid.v4()
    },
    name: "".concat(ruleName, " - exception list item"),
    namespace_type: namespaceType,
    tags: [],
    type: 'simple'
  };
};
/**
 * Returns the operator type, may not need this if using io-ts types
 *
 * @param item a single ExceptionItem entry
 */

export var getOperatorType = function getOperatorType(item) {
  switch (item.type) {
    case 'match':
      return OperatorTypeEnum.MATCH;

    case 'match_any':
      return OperatorTypeEnum.MATCH_ANY;

    case 'wildcard':
      return OperatorTypeEnum.WILDCARD;

    case 'list':
      return OperatorTypeEnum.LIST;

    default:
      return OperatorTypeEnum.EXISTS;
  }
};
/**
 * Determines operator selection (is/is not/is one of, etc.)
 * Default operator is "is"
 *
 * @param item a single ExceptionItem entry
 */

export var getExceptionOperatorSelect = function getExceptionOperatorSelect(item) {
  if (item.type === 'nested') {
    return isOperator;
  } else {
    var operatorType = getOperatorType(item);
    var foundOperator = ALL_OPERATORS.find(function (operatorOption) {
      return item.operator === operatorOption.operator && operatorType === operatorOption.type;
    });
    return foundOperator != null ? foundOperator : isOperator;
  }
};
/**
 * Returns the fields corresponding value for an entry
 *
 * @param item a single ExceptionItem entry
 */

export var getEntryValue = function getEntryValue(item) {
  switch (item.type) {
    case OperatorTypeEnum.MATCH:
    case OperatorTypeEnum.MATCH_ANY:
    case OperatorTypeEnum.WILDCARD:
      return item.value;

    case OperatorTypeEnum.EXISTS:
      return undefined;

    case OperatorTypeEnum.LIST:
      return item.list.id;

    default:
      return undefined;
  }
};
/**
 * Determines whether an entire entry, exception item, or entry within a nested
 * entry needs to be removed
 *
 * @param exceptionItem
 * @param entryIndex index of given entry, for nested entries, this will correspond
 * to their parent index
 * @param nestedEntryIndex index of nested entry
 *
 */

export var getUpdatedEntriesOnDelete = function getUpdatedEntriesOnDelete(exceptionItem, entryIndex, nestedParentIndex) {
  var itemOfInterest = exceptionItem.entries[nestedParentIndex != null ? nestedParentIndex : entryIndex];

  if (nestedParentIndex != null && itemOfInterest.type === OperatorTypeEnum.NESTED) {
    var updatedEntryEntries = [].concat(_toConsumableArray(itemOfInterest.entries.slice(0, entryIndex)), _toConsumableArray(itemOfInterest.entries.slice(entryIndex + 1)));

    if (updatedEntryEntries.length === 0) {
      return _objectSpread(_objectSpread({}, exceptionItem), {}, {
        entries: [].concat(_toConsumableArray(exceptionItem.entries.slice(0, nestedParentIndex)), _toConsumableArray(exceptionItem.entries.slice(nestedParentIndex + 1)))
      });
    } else {
      var field = itemOfInterest.field;
      var updatedItemOfInterest = {
        entries: updatedEntryEntries,
        field: field,
        id: itemOfInterest.id != null ? itemOfInterest.id : "".concat(entryIndex),
        type: OperatorTypeEnum.NESTED
      };
      return _objectSpread(_objectSpread({}, exceptionItem), {}, {
        entries: [].concat(_toConsumableArray(exceptionItem.entries.slice(0, nestedParentIndex)), [updatedItemOfInterest], _toConsumableArray(exceptionItem.entries.slice(nestedParentIndex + 1)))
      });
    }
  } else {
    return _objectSpread(_objectSpread({}, exceptionItem), {}, {
      entries: [].concat(_toConsumableArray(exceptionItem.entries.slice(0, entryIndex)), _toConsumableArray(exceptionItem.entries.slice(entryIndex + 1)))
    });
  }
};
/**
 * Returns filtered index patterns based on the field - if a user selects to
 * add nested entry, should only show nested fields, if item is the parent
 * field of a nested entry, we only display the parent field
 *
 * @param patterns DataViewBase containing available fields on rule index
 * @param item exception item entry
 * set to add a nested field
 */

export var getFilteredIndexPatterns = function getFilteredIndexPatterns(patterns, item, type, preFilter, osTypes) {
  var indexPatterns = preFilter != null ? preFilter(patterns, type, osTypes) : patterns;

  if (item.nested === 'child' && item.parent != null) {
    // when user has selected a nested entry, only fields with the common parent are shown
    return _objectSpread(_objectSpread({}, indexPatterns), {}, {
      fields: indexPatterns.fields.filter(function (indexField) {
        var subTypeNested = getDataViewFieldSubtypeNested(indexField);
        var fieldHasCommonParentPath = subTypeNested && item.parent != null && subTypeNested.nested.path === item.parent.parent.field;
        return fieldHasCommonParentPath;
      }).map(function (f) {
        var _f$name$split$slice = f.name.split('.').slice(-1),
            _f$name$split$slice2 = _slicedToArray(_f$name$split$slice, 1),
            fieldNameWithoutParentPath = _f$name$split$slice2[0];

        return _objectSpread(_objectSpread({}, f), {}, {
          name: fieldNameWithoutParentPath
        });
      })
    });
  } else if (item.nested === 'parent' && item.field != null) {
    // when user has selected a nested entry, right above it we show the common parent
    return _objectSpread(_objectSpread({}, indexPatterns), {}, {
      fields: [item.field]
    });
  } else if (item.nested === 'parent' && item.field == null) {
    // when user selects to add a nested entry, only nested fields are shown as options
    return _objectSpread(_objectSpread({}, indexPatterns), {}, {
      fields: indexPatterns.fields.filter(function (field) {
        return isDataViewFieldSubtypeNested(field);
      })
    });
  } else {
    return indexPatterns;
  }
};
/**
 * Determines proper entry update when user selects new field
 *
 * @param item - current exception item entry values
 * @param newField - newly selected field
 *
 */

export var getEntryOnFieldChange = function getEntryOnFieldChange(item, newField) {
  var parent = item.parent,
      entryIndex = item.entryIndex,
      nested = item.nested;
  var newChildFieldValue = newField != null ? newField.name.split('.').slice(-1)[0] : '';

  if (nested === 'parent') {
    // For nested entries, when user first selects to add a nested
    // entry, they first see a row similar to what is shown for when
    // a user selects "exists", as soon as they make a selection
    // we can now identify the 'parent' and 'child' this is where
    // we first convert the entry into type "nested"
    var subTypeNested = getDataViewFieldSubtypeNested(newField);
    var newParentFieldValue = (subTypeNested === null || subTypeNested === void 0 ? void 0 : subTypeNested.nested.path) || '';
    return {
      index: entryIndex,
      updatedEntry: {
        entries: [addIdToItem({
          field: newChildFieldValue != null ? newChildFieldValue : '',
          operator: isOperator.operator,
          type: OperatorTypeEnum.MATCH,
          value: ''
        })],
        field: newParentFieldValue,
        id: item.id,
        type: OperatorTypeEnum.NESTED
      }
    };
  } else if (nested === 'child' && parent != null) {
    return {
      index: parent.parentIndex,
      updatedEntry: _objectSpread(_objectSpread({}, parent.parent), {}, {
        entries: [].concat(_toConsumableArray(parent.parent.entries.slice(0, entryIndex)), [{
          field: newChildFieldValue != null ? newChildFieldValue : '',
          id: item.id,
          operator: isOperator.operator,
          type: OperatorTypeEnum.MATCH,
          value: ''
        }], _toConsumableArray(parent.parent.entries.slice(entryIndex + 1)))
      })
    };
  } else {
    return {
      index: entryIndex,
      updatedEntry: {
        field: newField != null ? newField.name : '',
        id: item.id,
        operator: isOperator.operator,
        type: OperatorTypeEnum.MATCH,
        value: ''
      }
    };
  }
};
/**
 * Determines proper entry update when user updates value
 * when operator is of type "list"
 *
 * @param item - current exception item entry values
 * @param newField - newly selected list
 *
 */

export var getEntryOnListChange = function getEntryOnListChange(item, newField) {
  var entryIndex = item.entryIndex,
      field = item.field,
      operator = item.operator;
  var id = newField.id,
      type = newField.type;
  return {
    index: entryIndex,
    updatedEntry: {
      field: field != null ? field.name : '',
      id: item.id,
      list: {
        id: id,
        type: type
      },
      operator: operator.operator,
      type: OperatorTypeEnum.LIST
    }
  };
};
/**
 * Determines proper entry update when user updates value
 * when operator is of type "match_any"
 *
 * @param item - current exception item entry values
 * @param newField - newly entered value
 *
 */

export var getEntryOnMatchAnyChange = function getEntryOnMatchAnyChange(item, newField) {
  var nested = item.nested,
      parent = item.parent,
      entryIndex = item.entryIndex,
      field = item.field,
      operator = item.operator;

  if (nested != null && parent != null) {
    var fieldName = field != null ? field.name.split('.').slice(-1)[0] : '';
    return {
      index: parent.parentIndex,
      updatedEntry: _objectSpread(_objectSpread({}, parent.parent), {}, {
        entries: [].concat(_toConsumableArray(parent.parent.entries.slice(0, entryIndex)), [{
          field: fieldName,
          id: item.id,
          operator: operator.operator,
          type: OperatorTypeEnum.MATCH_ANY,
          value: newField
        }], _toConsumableArray(parent.parent.entries.slice(entryIndex + 1)))
      })
    };
  } else {
    return {
      index: entryIndex,
      updatedEntry: {
        field: field != null ? field.name : '',
        id: item.id,
        operator: operator.operator,
        type: OperatorTypeEnum.MATCH_ANY,
        value: newField
      }
    };
  }
};
/**
 * Determines proper entry update when user updates value
 * when operator is of type "match"
 *
 * @param item - current exception item entry values
 * @param newField - newly entered value
 *
 */

export var getEntryOnMatchChange = function getEntryOnMatchChange(item, newField) {
  var nested = item.nested,
      parent = item.parent,
      entryIndex = item.entryIndex,
      field = item.field,
      operator = item.operator;

  if (nested != null && parent != null) {
    var fieldName = field != null ? field.name.split('.').slice(-1)[0] : '';
    return {
      index: parent.parentIndex,
      updatedEntry: _objectSpread(_objectSpread({}, parent.parent), {}, {
        entries: [].concat(_toConsumableArray(parent.parent.entries.slice(0, entryIndex)), [{
          field: fieldName,
          id: item.id,
          operator: operator.operator,
          type: OperatorTypeEnum.MATCH,
          value: newField
        }], _toConsumableArray(parent.parent.entries.slice(entryIndex + 1)))
      })
    };
  } else {
    return {
      index: entryIndex,
      updatedEntry: {
        field: field != null ? field.name : '',
        id: item.id,
        operator: operator.operator,
        type: OperatorTypeEnum.MATCH,
        value: newField
      }
    };
  }
};
/**
 * Determines proper entry update when user updates value
 * when operator is of type "wildcard"
 *
 * @param item - current exception item entry values
 * @param newField - newly entered value
 *
 */

export var getEntryOnWildcardChange = function getEntryOnWildcardChange(item, newField) {
  var nested = item.nested,
      parent = item.parent,
      entryIndex = item.entryIndex,
      field = item.field,
      operator = item.operator;

  if (nested != null && parent != null) {
    var fieldName = field != null ? field.name.split('.').slice(-1)[0] : '';
    return {
      index: parent.parentIndex,
      updatedEntry: _objectSpread(_objectSpread({}, parent.parent), {}, {
        entries: [].concat(_toConsumableArray(parent.parent.entries.slice(0, entryIndex)), [{
          field: fieldName,
          id: item.id,
          operator: operator.operator,
          type: OperatorTypeEnum.WILDCARD,
          value: newField
        }], _toConsumableArray(parent.parent.entries.slice(entryIndex + 1)))
      })
    };
  } else {
    return {
      index: entryIndex,
      updatedEntry: {
        field: field != null ? field.name : '',
        id: item.id,
        operator: operator.operator,
        type: OperatorTypeEnum.WILDCARD,
        value: newField
      }
    };
  }
};
/**
 * On operator change, determines whether value needs to be cleared or not
 *
 * @param field
 * @param selectedOperator
 * @param currentEntry
 *
 */

export var getEntryFromOperator = function getEntryFromOperator(selectedOperator, currentEntry) {
  var isSameOperatorType = currentEntry.operator.type === selectedOperator.type;
  var fieldValue = currentEntry.field != null ? currentEntry.field.name : '';

  switch (selectedOperator.type) {
    case 'match':
      return {
        field: fieldValue,
        id: currentEntry.id,
        operator: selectedOperator.operator,
        type: OperatorTypeEnum.MATCH,
        value: isSameOperatorType && typeof currentEntry.value === 'string' ? currentEntry.value : ''
      };

    case 'match_any':
      return {
        field: fieldValue,
        id: currentEntry.id,
        operator: selectedOperator.operator,
        type: OperatorTypeEnum.MATCH_ANY,
        value: isSameOperatorType && Array.isArray(currentEntry.value) ? currentEntry.value : []
      };

    case 'list':
      return {
        field: fieldValue,
        id: currentEntry.id,
        list: {
          id: '',
          type: 'ip'
        },
        operator: selectedOperator.operator,
        type: OperatorTypeEnum.LIST
      };

    case 'wildcard':
      return {
        field: fieldValue,
        id: currentEntry.id,
        operator: selectedOperator.operator,
        type: OperatorTypeEnum.WILDCARD,
        value: isSameOperatorType && typeof currentEntry.value === 'string' ? currentEntry.value : ''
      };

    default:
      return {
        field: fieldValue,
        id: currentEntry.id,
        operator: selectedOperator.operator,
        type: OperatorTypeEnum.EXISTS
      };
  }
};
/**
 * Determines proper entry update when user selects new operator
 *
 * @param item - current exception item entry values
 * @param newOperator - newly selected operator
 *
 */

export var getEntryOnOperatorChange = function getEntryOnOperatorChange(item, newOperator) {
  var parent = item.parent,
      entryIndex = item.entryIndex,
      field = item.field,
      nested = item.nested;
  var newEntry = getEntryFromOperator(newOperator, item);

  if (!entriesList.is(newEntry) && nested != null && parent != null) {
    return {
      index: parent.parentIndex,
      updatedEntry: _objectSpread(_objectSpread({}, parent.parent), {}, {
        entries: [].concat(_toConsumableArray(parent.parent.entries.slice(0, entryIndex)), [_objectSpread(_objectSpread({}, newEntry), {}, {
          field: field != null ? field.name.split('.').slice(-1)[0] : ''
        })], _toConsumableArray(parent.parent.entries.slice(entryIndex + 1)))
      })
    };
  } else {
    return {
      index: entryIndex,
      updatedEntry: newEntry
    };
  }
};

var fieldSupportsMatches = function fieldSupportsMatches(field) {
  return field.type === 'string';
};
/**
 * Determines which operators to make available
 *
 * @param item
 * @param listType
 * @param isBoolean
 * @param includeValueListOperators whether or not to include the 'is in list' and 'is not in list' operators
 */


export var getOperatorOptions = function getOperatorOptions(item, listType, isBoolean) {
  var includeValueListOperators = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : true;

  if (item.nested === 'parent' || item.field == null) {
    return [isOperator];
  } else if (item.nested != null && listType === 'endpoint' || listType === 'endpoint') {
    return isBoolean ? [isOperator] : [isOperator, isOneOfOperator];
  } else if (item.nested != null && listType === 'detection') {
    return isBoolean ? [isOperator, existsOperator] : [isOperator, isOneOfOperator, existsOperator];
  } else if (isBoolean) {
    return [isOperator, isNotOperator, existsOperator, doesNotExistOperator];
  } else if (!includeValueListOperators) {
    return fieldSupportsMatches(item.field) ? EXCEPTION_OPERATORS_SANS_LISTS : [isOperator, isNotOperator, isOneOfOperator, isNotOneOfOperator, existsOperator, doesNotExistOperator];
  } else {
    return listType === 'detection' ? fieldSupportsMatches(item.field) ? DETECTION_ENGINE_EXCEPTION_OPERATORS : [isOperator, isNotOperator, isOneOfOperator, isNotOneOfOperator, existsOperator, doesNotExistOperator, isInListOperator, isNotInListOperator] : ALL_OPERATORS;
  }
};
/**
 * Fields of type 'text' do not generate autocomplete values, we want
 * to find it's corresponding keyword type (if available) which does
 * generate autocomplete values
 *
 * @param fields DataViewFieldBase fields
 * @param selectedField the field name that was selected
 * @param isTextType we only want a corresponding keyword field if
 * the selected field is of type 'text'
 *
 */

export var getCorrespondingKeywordField = function getCorrespondingKeywordField(_ref2) {
  var fields = _ref2.fields,
      selectedField = _ref2.selectedField;
  var selectedFieldBits = selectedField != null && selectedField !== '' ? selectedField.split('.') : [];
  var selectedFieldIsTextType = selectedFieldBits.slice(-1)[0] === 'text';

  if (selectedFieldIsTextType && selectedFieldBits.length > 0) {
    var keywordField = selectedFieldBits.slice(0, selectedFieldBits.length - 1).join('.');

    var _fields$filter = fields.filter(function (_ref3) {
      var name = _ref3.name;
      return keywordField !== '' && keywordField === name;
    }),
        _fields$filter2 = _slicedToArray(_fields$filter, 1),
        foundKeywordField = _fields$filter2[0];

    return foundKeywordField;
  }

  return undefined;
};
/**
 * Formats the entry into one that is easily usable for the UI, most of the
 * complexity was introduced with nested fields
 *
 * @param patterns DataViewBase containing available fields on rule index
 * @param item exception item entry
 * @param itemIndex entry index
 * @param parent nested entries hold copy of their parent for use in various logic
 * @param parentIndex corresponds to the entry index, this might seem obvious, but
 * was added to ensure that nested items could be identified with their parent entry
 */

export var getFormattedBuilderEntry = function getFormattedBuilderEntry(indexPattern, item, itemIndex, parent, parentIndex) {
  var fields = indexPattern.fields;
  var field = parent != null ? "".concat(parent.field, ".").concat(item.field) : item.field;

  var _fields$filter3 = fields.filter(function (_ref4) {
    var name = _ref4.name;
    return field != null && field === name;
  }),
      _fields$filter4 = _slicedToArray(_fields$filter3, 1),
      foundField = _fields$filter4[0];

  var correspondingKeywordField = getCorrespondingKeywordField({
    fields: fields,
    selectedField: field
  });

  if (parent != null && parentIndex != null) {
    return {
      correspondingKeywordField: correspondingKeywordField,
      entryIndex: itemIndex,
      field: foundField != null ? _objectSpread(_objectSpread({}, foundField), {}, {
        name: foundField.name.split('.').slice(-1)[0]
      }) : foundField,
      id: item.id != null ? item.id : "".concat(itemIndex),
      nested: 'child',
      operator: getExceptionOperatorSelect(item),
      parent: {
        parent: parent,
        parentIndex: parentIndex
      },
      value: getEntryValue(item)
    };
  } else {
    return {
      correspondingKeywordField: correspondingKeywordField,
      entryIndex: itemIndex,
      field: foundField,
      id: item.id != null ? item.id : "".concat(itemIndex),
      nested: undefined,
      operator: getExceptionOperatorSelect(item),
      parent: undefined,
      value: getEntryValue(item)
    };
  }
};
/**
 * Formats the entries to be easily usable for the UI, most of the
 * complexity was introduced with nested fields
 *
 * @param patterns DataViewBase containing available fields on rule index
 * @param entries exception item entries
 * @param addNested boolean noting whether or not UI is currently
 * set to add a nested field
 * @param parent nested entries hold copy of their parent for use in various logic
 * @param parentIndex corresponds to the entry index, this might seem obvious, but
 * was added to ensure that nested items could be identified with their parent entry
 */

export var getFormattedBuilderEntries = function getFormattedBuilderEntries(indexPattern, entries, parent, parentIndex) {
  return entries.reduce(function (acc, item, index) {
    var isNewNestedEntry = item.type === 'nested' && item.entries.length === 0;

    if (item.type !== 'nested' && !isNewNestedEntry) {
      var newItemEntry = getFormattedBuilderEntry(indexPattern, item, index, parent, parentIndex);
      return [].concat(_toConsumableArray(acc), [newItemEntry]);
    } else {
      var parentEntry = {
        correspondingKeywordField: undefined,
        entryIndex: index,
        field: isNewNestedEntry ? undefined : // This type below is really a FieldSpec type from "src/plugins/data/common/index_patterns/fields/types.ts", we cast it here to keep using the DataViewFieldBase interface
        {
          aggregatable: false,
          esTypes: ['nested'],
          name: item.field != null ? item.field : '',
          searchable: false,
          type: 'string'
        },
        id: item.id != null ? item.id : "".concat(index),
        nested: 'parent',
        operator: isOperator,
        parent: undefined,
        value: undefined
      }; // User has selected to add a nested field, but not yet selected the field

      if (isNewNestedEntry) {
        return [].concat(_toConsumableArray(acc), [parentEntry]);
      }

      if (isEntryNested(item)) {
        var nestedItems = getFormattedBuilderEntries(indexPattern, item.entries, item, index);
        return [].concat(_toConsumableArray(acc), [parentEntry], _toConsumableArray(nestedItems));
      }

      return _toConsumableArray(acc);
    }
  }, []);
};
export var getDefaultEmptyEntry = function getDefaultEmptyEntry() {
  return {
    field: '',
    id: uuid.v4(),
    operator: OperatorEnum.INCLUDED,
    type: OperatorTypeEnum.MATCH,
    value: ''
  };
};
export var getDefaultNestedEmptyEntry = function getDefaultNestedEmptyEntry() {
  return {
    entries: [],
    field: '',
    id: uuid.v4(),
    type: OperatorTypeEnum.NESTED
  };
};
export var containsValueListEntry = function containsValueListEntry(items) {
  return items.some(function (item) {
    return item.entries.some(function (_ref5) {
      var type = _ref5.type;
      return type === OperatorTypeEnum.LIST;
    });
  });
};