"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.streamFactory = streamFactory;

var _stream = require("stream");

var zlib = _interopRequireWildcard(require("zlib"));

var _accept_compression = require("./accept_compression");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
// We need this otherwise Kibana server will crash with a 'ERR_METHOD_NOT_IMPLEMENTED' error.
class ResponseStream extends _stream.Stream.PassThrough {
  flush() {}

  _read() {}

}

const DELIMITER = '\n';

/**
 * Sets up a response stream with support for gzip compression depending on provided
 * request headers. Any non-string data pushed to the stream will be stream as NDJSON.
 *
 * @param headers - Request headers.
 * @returns An object with stream attributes and methods.
 */
function streamFactory(headers, logger) {
  let streamType;
  const isCompressed = (0, _accept_compression.acceptCompression)(headers);
  const stream = isCompressed ? zlib.createGzip() : new ResponseStream();

  function end() {
    stream.end();
  }

  function push(d) {
    if (d === undefined) {
      logger.error('Stream chunk must not be undefined.');
      return;
    } // Initialize the stream type with the first push to the stream,
    // otherwise check the integrity of the data to be pushed.


    if (streamType === undefined) {
      streamType = typeof d === 'string' ? 'string' : 'ndjson';
    } else if (streamType === 'string' && typeof d !== 'string') {
      logger.error('Must not push non-string chunks to a string based stream.');
      return;
    } else if (streamType === 'ndjson' && typeof d === 'string') {
      logger.error('Must not push raw string chunks to an NDJSON based stream.');
      return;
    }

    try {
      const line = typeof d !== 'string' ? `${JSON.stringify(d)}${DELIMITER}` : d;
      stream.write(line);
    } catch (e) {
      logger.error(`Could not serialize or stream data chunk: ${e.toString()}`);
      return;
    } // Calling .flush() on a compression stream will
    // make zlib return as much output as currently possible.


    if (isCompressed) {
      stream.flush();
    }
  }

  const responseWithHeaders = {
    body: stream,
    ...(isCompressed ? {
      headers: {
        'content-encoding': 'gzip'
      }
    } : {})
  };
  return {
    DELIMITER,
    end,
    push,
    responseWithHeaders
  };
}