"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runInAutoMode = void 0;

var _lodash = require("lodash");

var _constants = require("./constants");

var _stack_services = require("../common/stack_services");

var _utils = require("./utils");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const ACTION_RESPONSE_DELAY = 40_000;

const runInAutoMode = async ({
  log,
  flags: {
    username,
    password,
    asSuperuser,
    kibana,
    elastic,
    delay: _delay
  }
}) => {
  const runtimeServices = await (0, _stack_services.createRuntimeServices)({
    log,
    password: password,
    username: username,
    asSuperuser: asSuperuser,
    elasticsearchUrl: elastic,
    kibanaUrl: kibana
  });
  log.write(`  ${_constants.SUPPORTED_TOKENS}`);
  const delay = Number(_delay) || ACTION_RESPONSE_DELAY;

  do {
    await checkPendingActionsAndRespond(runtimeServices, {
      delay
    });
    await (0, _utils.sleep)(5_000);
  } while (true);
};

exports.runInAutoMode = runInAutoMode;

const checkPendingActionsAndRespond = async ({
  kbnClient,
  esClient,
  log
}, {
  delay = ACTION_RESPONSE_DELAY
} = {}) => {
  let hasMore = true;
  let nextPage = 1;

  try {
    while (hasMore) {
      const {
        data: actions
      } = await (0, _utils.fetchEndpointActionList)(kbnClient, {
        page: nextPage++,
        pageSize: 100
      });

      if (actions.length === 0) {
        hasMore = false;
      }

      for (const action of actions) {
        if (action.isCompleted === false) {
          if (Date.now() - new Date(action.startedAt).getTime() >= delay) {
            log.info(`[${new Date().toLocaleTimeString()}]: Responding to [${action.command}] action [id: ${action.id}] agent: [${action.agents.join(', ')}]`);
            const tokens = parseCommentTokens(getActionComment(action));
            log.verbose('tokens found in action:', tokens);
            const fleetResponse = await (0, _utils.sendFleetActionResponse)(esClient, action, {
              // If an Endpoint state token was found, then force the Fleet response to `success`
              // so that we can actually generate an endpoint response below.
              state: tokens.state ? 'success' : tokens.fleet.state
            }); // If not a fleet response error, then also sent the Endpoint Response

            if (!fleetResponse.error) {
              await (0, _utils.sendEndpointActionResponse)(esClient, action, {
                state: tokens.state
              });
            }
          }
        }
      }
    }
  } catch (e) {
    log.error(`${e.message}. Run with '--verbose' option to see more`);
    log.verbose(e);
  }
};

const parseCommentTokens = comment => {
  const response = {
    state: undefined,
    fleet: {
      state: undefined
    }
  };

  if (comment) {
    const findTokensRegExp = /(respond\.\S*=\S*)/gi;
    let matches;

    while ((matches = findTokensRegExp.exec(comment)) !== null) {
      const [key, value] = matches[0].toLowerCase().split('=').map(s => s.trim());
      (0, _lodash.set)(response, key.split('.').slice(1), value);
    }
  }

  return response;
};

const getActionComment = action => {
  var _action$comment;

  return (_action$comment = action.comment) !== null && _action$comment !== void 0 ? _action$comment : '';
};