"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createLiveQueryRoute = void 0;

var _lodash = require("lodash");

var _uuid = _interopRequireDefault(require("uuid"));

var _momentTimezone = _interopRequireDefault(require("moment-timezone"));

var _common = require("../../../../fleet/common");

var _parse_agent_groups = require("../../lib/parse_agent_groups");

var _route_validation = require("../../utils/build_validation/route_validation");

var _live_query = require("../../../common/schemas/routes/live_query");

var _types = require("../../../common/types");

var _constants = require("../../../common/constants");

var _utils = require("../pack/utils");

var _constants2 = require("../../lib/telemetry/constants");

var _utils2 = require("../saved_query/utils");

var _utils3 = require("../utils");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const createLiveQueryRoute = (router, osqueryContext) => {
  router.post({
    path: '/api/osquery/live_queries',
    validate: {
      body: (0, _route_validation.buildRouteValidation)(_live_query.createLiveQueryRequestBodySchema)
    }
  }, async (context, request, response) => {
    const coreContext = await context.core;
    const esClient = coreContext.elasticsearch.client.asInternalUser;
    const soClient = coreContext.savedObjects.client;
    const internalSavedObjectsClient = await (0, _utils3.getInternalSavedObjectsClient)(osqueryContext.getStartServices);
    const [coreStartServices] = await osqueryContext.getStartServices();
    let savedQueryId = request.body.saved_query_id;
    const {
      osquery: {
        writeLiveQueries,
        runSavedQueries
      }
    } = await coreStartServices.capabilities.resolveCapabilities(request);
    const isInvalid = !(writeLiveQueries || runSavedQueries && (request.body.saved_query_id || request.body.pack_id));

    if (isInvalid) {
      return response.forbidden();
    }

    if (request.body.saved_query_id && runSavedQueries) {
      const savedQueries = await soClient.find({
        type: _types.savedQuerySavedObjectType
      });
      const actualSavedQuery = savedQueries.saved_objects.find(savedQuery => savedQuery.id === request.body.saved_query_id);

      if (actualSavedQuery) {
        savedQueryId = actualSavedQuery.id;
      }
    } // eslint-disable-next-line @typescript-eslint/naming-convention


    const {
      agent_all,
      agent_ids,
      agent_platforms,
      agent_policy_ids
    } = request.body;
    const selectedAgents = await (0, _parse_agent_groups.parseAgentSelection)(internalSavedObjectsClient, osqueryContext, {
      agents: agent_ids,
      allAgentsSelected: !!agent_all,
      platformsSelected: agent_platforms,
      policiesSelected: agent_policy_ids
    });

    if (!selectedAgents.length) {
      return response.badRequest({
        body: new Error('No agents found for selection')
      });
    }

    try {
      var _osqueryContext$secur, _packSO, _packSO$attributes, _packSO2, _osqueryContext$servi;

      const currentUser = await ((_osqueryContext$secur = osqueryContext.security.authc.getCurrentUser(request)) === null || _osqueryContext$secur === void 0 ? void 0 : _osqueryContext$secur.username);
      let packSO;

      if (request.body.pack_id) {
        packSO = await soClient.get(_types.packSavedObjectType, request.body.pack_id);
      }

      const osqueryAction = {
        action_id: _uuid.default.v4(),
        '@timestamp': (0, _momentTimezone.default)().toISOString(),
        expiration: (0, _momentTimezone.default)().add(5, 'minutes').toISOString(),
        type: 'INPUT_ACTION',
        input_type: 'osquery',
        alert_ids: request.body.alert_ids,
        event_ids: request.body.event_ids,
        case_ids: request.body.case_ids,
        agent_ids: request.body.agent_ids,
        agent_all: request.body.agent_all,
        agent_platforms: request.body.agent_platforms,
        agent_policy_ids: request.body.agent_policy_ids,
        agents: selectedAgents,
        user_id: currentUser,
        metadata: request.body.metadata,
        pack_id: request.body.pack_id,
        pack_name: (_packSO = packSO) === null || _packSO === void 0 ? void 0 : (_packSO$attributes = _packSO.attributes) === null || _packSO$attributes === void 0 ? void 0 : _packSO$attributes.name,
        pack_prebuilt: request.body.pack_id ? !!(0, _lodash.some)((_packSO2 = packSO) === null || _packSO2 === void 0 ? void 0 : _packSO2.references, ['type', 'osquery-pack-asset']) : undefined,
        queries: packSO ? (0, _lodash.map)((0, _utils.convertSOQueriesToPack)(packSO.attributes.queries), (packQuery, packQueryId) => (0, _lodash.pickBy)({
          action_id: _uuid.default.v4(),
          id: packQueryId,
          query: packQuery.query,
          ecs_mapping: packQuery.ecs_mapping,
          version: packQuery.version,
          platform: packQuery.platform,
          agents: selectedAgents
        }, value => !(0, _lodash.isEmpty)(value))) : [(0, _lodash.pickBy)({
          action_id: _uuid.default.v4(),
          id: _uuid.default.v4(),
          query: request.body.query,
          saved_query_id: savedQueryId,
          saved_query_prebuilt: savedQueryId ? await (0, _utils2.isSavedQueryPrebuilt)((_osqueryContext$servi = osqueryContext.service.getPackageService()) === null || _osqueryContext$servi === void 0 ? void 0 : _osqueryContext$servi.asInternalUser, savedQueryId) : undefined,
          ecs_mapping: request.body.ecs_mapping,
          agents: selectedAgents
        }, value => !(0, _lodash.isEmpty)(value))]
      };
      const fleetActions = (0, _lodash.map)(osqueryAction.queries, query => ({
        action_id: query.action_id,
        '@timestamp': (0, _momentTimezone.default)().toISOString(),
        expiration: (0, _momentTimezone.default)().add(5, 'minutes').toISOString(),
        type: 'INPUT_ACTION',
        input_type: 'osquery',
        agents: query.agents,
        user_id: currentUser,
        data: (0, _lodash.pick)(query, ['id', 'query', 'ecs_mapping', 'version', 'platform'])
      }));
      await esClient.bulk({
        refresh: 'wait_for',
        body: (0, _lodash.flatten)(fleetActions.map(action => [{
          index: {
            _index: _common.AGENT_ACTIONS_INDEX
          }
        }, action]))
      });
      const actionsComponentTemplateExists = await esClient.indices.exists({
        index: `${_constants.ACTIONS_INDEX}*`
      });

      if (actionsComponentTemplateExists) {
        await esClient.bulk({
          refresh: 'wait_for',
          body: [{
            index: {
              _index: `${_constants.ACTIONS_INDEX}-default`
            }
          }, osqueryAction]
        });
      }

      osqueryContext.telemetryEventsSender.reportEvent(_constants2.TELEMETRY_EBT_LIVE_QUERY_EVENT, { ...(0, _lodash.omit)(osqueryAction, ['type', 'input_type', 'user_id']),
        agents: osqueryAction.agents.length
      });
      return response.ok({
        body: {
          data: osqueryAction
        }
      });
    } catch (error) {
      return response.customError({
        statusCode: 500,
        body: new Error(`Error occurred while processing ${error}`)
      });
    }
  });
};

exports.createLiveQueryRoute = createLiveQueryRoute;