"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateAgentTags = updateAgentTags;

var _lodash = require("lodash");

var _errors = require("../../errors");

var _crud = require("./crud");

var _helpers = require("./helpers");

var _filter_hosted_agents = require("./filter_hosted_agents");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
function isMgetDoc(doc) {
  return Boolean(doc && 'found' in doc);
}

async function updateAgentTags(soClient, esClient, options, tagsToAdd, tagsToRemove) {
  const outgoingErrors = {};
  const givenAgents = [];

  if ('agentIds' in options) {
    const givenAgentsResults = await (0, _crud.getAgentDocuments)(esClient, options.agentIds);

    for (const agentResult of givenAgentsResults) {
      if (isMgetDoc(agentResult) && agentResult.found === false) {
        outgoingErrors[agentResult._id] = new _errors.AgentReassignmentError(`Cannot find agent ${agentResult._id}`);
      } else {
        givenAgents.push((0, _helpers.searchHitToAgent)(agentResult));
      }
    }
  } else if ('kuery' in options) {
    return await (0, _crud.processAgentsInBatches)(esClient, {
      kuery: options.kuery,
      showInactive: true,
      batchSize: options.batchSize
    }, async (agents, skipSuccess) => await updateTagsBatch(soClient, esClient, agents, outgoingErrors, tagsToAdd, tagsToRemove, undefined, skipSuccess));
  }

  return await updateTagsBatch(soClient, esClient, givenAgents, outgoingErrors, tagsToAdd, tagsToRemove, 'agentIds' in options ? options.agentIds : undefined);
}

async function updateTagsBatch(soClient, esClient, givenAgents, outgoingErrors, tagsToAdd, tagsToRemove, agentIds, skipSuccess) {
  const errors = { ...outgoingErrors
  };
  const filteredAgents = await (0, _filter_hosted_agents.filterHostedPolicies)(soClient, givenAgents, errors, `Cannot modify tags on a hosted agent`);

  const getNewTags = agent => {
    var _agent$tags;

    const existingTags = (_agent$tags = agent.tags) !== null && _agent$tags !== void 0 ? _agent$tags : [];

    if (tagsToAdd.length === 1 && tagsToRemove.length === 1) {
      const removableTagIndex = existingTags.indexOf(tagsToRemove[0]);

      if (removableTagIndex > -1) {
        const newTags = (0, _lodash.uniq)([...existingTags.slice(0, removableTagIndex), tagsToAdd[0], ...existingTags.slice(removableTagIndex + 1)]);
        return newTags;
      }
    }

    return (0, _lodash.uniq)((0, _lodash.difference)(existingTags, tagsToRemove).concat(tagsToAdd));
  };

  await (0, _crud.bulkUpdateAgents)(esClient, filteredAgents.map(agent => ({
    agentId: agent.id,
    data: {
      tags: getNewTags(agent)
    }
  })));
  return {
    items: (0, _crud.errorsToResults)(filteredAgents, errors, agentIds, skipSuccess)
  };
}