"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mapIndexStats = exports.fetchIndices = exports.fetchIndexCounts = void 0;

var _configSchema = require("@kbn/config-schema");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const mapIndexStats = (indexData, indexStats, indexName) => {
  var _indexStats$total$sto, _indexStats$total, _indexStats$total$sto2, _indexStats$total$doc, _indexStats$total2, _indexStats$total2$do, _indexStats$total$doc2, _indexStats$total3, _indexStats$total3$do;

  const aliases = Object.keys(indexData.aliases);
  const sizeInBytes = new _configSchema.ByteSizeValue((_indexStats$total$sto = indexStats === null || indexStats === void 0 ? void 0 : (_indexStats$total = indexStats.total) === null || _indexStats$total === void 0 ? void 0 : (_indexStats$total$sto2 = _indexStats$total.store) === null || _indexStats$total$sto2 === void 0 ? void 0 : _indexStats$total$sto2.size_in_bytes) !== null && _indexStats$total$sto !== void 0 ? _indexStats$total$sto : 0).toString();
  const docCount = (_indexStats$total$doc = indexStats === null || indexStats === void 0 ? void 0 : (_indexStats$total2 = indexStats.total) === null || _indexStats$total2 === void 0 ? void 0 : (_indexStats$total2$do = _indexStats$total2.docs) === null || _indexStats$total2$do === void 0 ? void 0 : _indexStats$total2$do.count) !== null && _indexStats$total$doc !== void 0 ? _indexStats$total$doc : 0;
  const docDeleted = (_indexStats$total$doc2 = indexStats === null || indexStats === void 0 ? void 0 : (_indexStats$total3 = indexStats.total) === null || _indexStats$total3 === void 0 ? void 0 : (_indexStats$total3$do = _indexStats$total3.docs) === null || _indexStats$total3$do === void 0 ? void 0 : _indexStats$total3$do.deleted) !== null && _indexStats$total$doc2 !== void 0 ? _indexStats$total$doc2 : 0;
  const total = {
    docs: {
      count: docCount,
      deleted: docDeleted
    },
    store: {
      size_in_bytes: sizeInBytes
    }
  };
  return {
    aliases,
    health: indexStats === null || indexStats === void 0 ? void 0 : indexStats.health,
    name: indexName,
    status: indexStats === null || indexStats === void 0 ? void 0 : indexStats.status,
    total,
    uuid: indexStats === null || indexStats === void 0 ? void 0 : indexStats.uuid
  };
};

exports.mapIndexStats = mapIndexStats;

const fetchIndexCounts = async (client, indicesNames) => {
  // TODO: is there way to batch this? Passing multiple index names or a pattern still returns a singular count
  const countPromises = indicesNames.map(async indexName => {
    const {
      count
    } = await client.asCurrentUser.count({
      index: indexName
    });
    return {
      [indexName]: count
    };
  });
  const indexCountArray = await Promise.all(countPromises);
  return indexCountArray.reduce((acc, current) => ({ ...acc,
    ...current
  }), {});
};

exports.fetchIndexCounts = fetchIndexCounts;

const fetchIndices = async (client, indexPattern, returnHiddenIndices, includeAliases, alwaysShowSearchPattern) => {
  // This call retrieves alias and settings information about indices
  // If we provide an override pattern with alwaysShowSearchPattern we get everything and filter out hiddens.
  const expandWildcards = returnHiddenIndices || alwaysShowSearchPattern ? ['hidden', 'all'] : ['open'];
  const totalIndices = await client.asCurrentUser.indices.get({
    expand_wildcards: expandWildcards,
    // for better performance only compute aliases and settings of indices but not mappings
    features: ['aliases', 'settings'],
    // only get specified index properties from ES to keep the response under 536MB
    // node.js string length limit: https://github.com/nodejs/node/issues/33960
    filter_path: ['*.aliases', '*.settings.index.hidden'],
    index: indexPattern
  }); // Index names that with one of their aliases match with the alwaysShowSearchPattern

  const alwaysShowPatternMatches = new Set();
  const indexAndAliasNames = Object.keys(totalIndices).reduce((accum, indexName) => {
    accum.push(indexName);

    if (includeAliases) {
      const aliases = Object.keys(totalIndices[indexName].aliases);
      aliases.forEach(alias => {
        accum.push(alias); // Add indexName to the set if an alias matches the pattern

        if (alwaysShowSearchPattern && alias.startsWith(alwaysShowSearchPattern)) {
          alwaysShowPatternMatches.add(indexName);
        }
      });
    }

    return accum;
  }, []);
  const indicesNames = returnHiddenIndices ? Object.keys(totalIndices) : Object.keys(totalIndices).filter(indexName => {
    var _totalIndices$indexNa, _totalIndices$indexNa2, _totalIndices$indexNa3;

    return !(((_totalIndices$indexNa = totalIndices[indexName]) === null || _totalIndices$indexNa === void 0 ? void 0 : (_totalIndices$indexNa2 = _totalIndices$indexNa.settings) === null || _totalIndices$indexNa2 === void 0 ? void 0 : (_totalIndices$indexNa3 = _totalIndices$indexNa2.index) === null || _totalIndices$indexNa3 === void 0 ? void 0 : _totalIndices$indexNa3.hidden) === 'true');
  });

  if (indicesNames.length === 0) {
    return [];
  }

  const {
    indices: indicesStats = {}
  } = await client.asCurrentUser.indices.stats({
    expand_wildcards: expandWildcards,
    index: indexPattern,
    metric: ['docs', 'store']
  }); // TODO: make multiple batched requests if indicesNames.length > SOMETHING

  const {
    index: indexPrivileges
  } = await client.asCurrentUser.security.hasPrivileges({
    index: [{
      names: indexAndAliasNames,
      privileges: ['read', 'manage']
    }]
  });
  const indexCounts = await fetchIndexCounts(client, indexAndAliasNames); // Index data to show even if they are hidden, set by alwaysShowSearchPattern

  const alwaysShowIndices = alwaysShowSearchPattern ? Array.from(alwaysShowPatternMatches).map(indexName => {
    const indexData = totalIndices[indexName];
    const indexStats = indicesStats[indexName];
    return mapIndexStats(indexData, indexStats, indexName);
  }).flatMap(({
    name,
    aliases,
    ...indexData
  }) => {
    const indicesAndAliases = [];

    if (includeAliases) {
      aliases.forEach(alias => {
        if (alias.startsWith(alwaysShowSearchPattern)) {
          var _indexCounts$alias;

          indicesAndAliases.push({
            alias: true,
            count: (_indexCounts$alias = indexCounts[alias]) !== null && _indexCounts$alias !== void 0 ? _indexCounts$alias : 0,
            name: alias,
            privileges: {
              manage: false,
              read: false,
              ...indexPrivileges[name]
            },
            ...indexData
          });
        }
      });
    }

    return indicesAndAliases;
  }) : [];
  const regularIndexData = indicesNames.map(indexName => {
    const indexData = totalIndices[indexName];
    const indexStats = indicesStats[indexName];
    return mapIndexStats(indexData, indexStats, indexName);
  }).flatMap(({
    name,
    aliases,
    ...indexData
  }) => {
    var _indexCounts$name;

    // expand aliases and add to results
    const indicesAndAliases = [];
    indicesAndAliases.push({
      alias: false,
      count: (_indexCounts$name = indexCounts[name]) !== null && _indexCounts$name !== void 0 ? _indexCounts$name : 0,
      name,
      privileges: {
        manage: false,
        read: false,
        ...indexPrivileges[name]
      },
      ...indexData
    });

    if (includeAliases) {
      aliases.forEach(alias => {
        var _indexCounts$alias2;

        indicesAndAliases.push({
          alias: true,
          count: (_indexCounts$alias2 = indexCounts[alias]) !== null && _indexCounts$alias2 !== void 0 ? _indexCounts$alias2 : 0,
          name: alias,
          privileges: {
            manage: false,
            read: false,
            ...indexPrivileges[name]
          },
          ...indexData
        });
      });
    }

    return indicesAndAliases;
  });
  const indexNamesAlreadyIncluded = regularIndexData.map(({
    name
  }) => name);
  const indexNamesToInclude = alwaysShowIndices.map(({
    name
  }) => name).filter(name => !indexNamesAlreadyIncluded.includes(name));
  const itemsToInclude = alwaysShowIndices.filter(({
    name
  }) => indexNamesToInclude.includes(name));
  const indicesData = alwaysShowSearchPattern ? [...regularIndexData, ...itemsToInclude] : regularIndexData;
  return indicesData.filter(({
    name
  }, index, array) => // make list of aliases unique since we add an alias per index above
  // and aliases can point to multiple indices
  array.findIndex(engineData => engineData.name === name) === index);
};

exports.fetchIndices = fetchIndices;