"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformOsType = exports.getBaseNestedClause = exports.getBaseMatchAnyClause = exports.createOrClauses = exports.createInnerAndClauses = exports.chunkExceptions = exports.buildNestedClause = exports.buildMatchWildcardClause = exports.buildMatchClause = exports.buildMatchAnyClause = exports.buildExistsClause = exports.buildExclusionClause = exports.buildExceptionItemFilterWithOsType = exports.buildExceptionItemFilter = exports.buildExceptionFilter = void 0;

var _fp = require("lodash/fp");

var _securitysolutionIoTsListTypes = require("@kbn/securitysolution-io-ts-list-types");

var _has_large_value_list = require("../has_large_value_list");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
const chunkExceptions = (exceptions, chunkSize) => {
  return (0, _fp.chunk)(chunkSize, exceptions);
};
/**
 * Transforms the os_type into a regular filter as if the user had created it
 * from the fields for the next state of transforms which will create the elastic filters
 * from it.
 *
 * Note: We use two types of fields, the "host.os.type" and "host.os.name.caseless"
 * The endpoint/endgame agent has been using "host.os.name.caseless" as the same value as the ECS
 * value of "host.os.type" where the auditbeat, winlogbeat, etc... (other agents) are all using
 * "host.os.type". In order to be compatible with both, I create an "OR" between these two data types
 * where if either has a match then we will exclude it as part of the match. This should also be
 * forwards compatible for endpoints/endgame agents when/if they upgrade to using "host.os.type"
 * rather than using "host.os.name.caseless" values.
 *
 * Also we create another "OR" from the osType names so that if there are multiples such as ['windows', 'linux']
 * this will exclude anything with either 'windows' or with 'linux'
 * @param osTypes The os_type array from the REST interface that is an array such as ['windows', 'linux']
 * @param entries The entries to join the OR's with before the elastic filter change out
 */


exports.chunkExceptions = chunkExceptions;

const transformOsType = (osTypes, entries) => {
  const hostTypeTransformed = osTypes.map(osType => {
    return [{
      field: 'host.os.type',
      operator: 'included',
      type: 'match',
      value: osType
    }, ...entries];
  });
  const caseLessTransformed = osTypes.map(osType => {
    return [{
      field: 'host.os.name.caseless',
      operator: 'included',
      type: 'match',
      value: osType
    }, ...entries];
  });
  return [...hostTypeTransformed, ...caseLessTransformed];
};
/**
 * This builds an exception item filter with the os type
 * @param osTypes The os_type array from the REST interface that is an array such as ['windows', 'linux']
 * @param entries The entries to join the OR's with before the elastic filter change out
 */


exports.transformOsType = transformOsType;

const buildExceptionItemFilterWithOsType = (osTypes, entries) => {
  const entriesWithOsTypes = transformOsType(osTypes, entries);
  return entriesWithOsTypes.map(entryWithOsType => {
    return {
      bool: {
        filter: entryWithOsType.map(entry => createInnerAndClauses(entry))
      }
    };
  });
};

exports.buildExceptionItemFilterWithOsType = buildExceptionItemFilterWithOsType;

const buildExceptionItemFilter = exceptionItem => {
  const {
    entries,
    os_types: osTypes
  } = exceptionItem;

  if (osTypes != null && osTypes.length > 0) {
    return buildExceptionItemFilterWithOsType(osTypes, entries);
  } else {
    if (entries.length === 1) {
      return [createInnerAndClauses(entries[0])];
    } else {
      return [{
        bool: {
          filter: entries.map(entry => createInnerAndClauses(entry))
        }
      }];
    }
  }
};

exports.buildExceptionItemFilter = buildExceptionItemFilter;

const createOrClauses = exceptionItems => {
  return exceptionItems.flatMap(exceptionItem => buildExceptionItemFilter(exceptionItem));
};

exports.createOrClauses = createOrClauses;

const buildExceptionFilter = ({
  lists,
  excludeExceptions,
  chunkSize,
  alias = null
}) => {
  // Remove exception items with large value lists. These are evaluated
  // elsewhere for the moment being.
  const exceptionsWithoutLargeValueLists = lists.filter(item => !(0, _has_large_value_list.hasLargeValueList)(item.entries));
  const exceptionFilter = {
    meta: {
      alias,
      disabled: false,
      negate: excludeExceptions
    },
    query: {
      bool: {
        should: undefined
      }
    }
  };

  if (exceptionsWithoutLargeValueLists.length === 0) {
    return undefined;
  } else if (exceptionsWithoutLargeValueLists.length <= chunkSize) {
    const clause = createOrClauses(exceptionsWithoutLargeValueLists);
    exceptionFilter.query.bool.should = clause;
    return exceptionFilter;
  } else {
    const chunks = chunkExceptions(exceptionsWithoutLargeValueLists, chunkSize);
    const filters = chunks.map(exceptionsChunk => {
      const orClauses = createOrClauses(exceptionsChunk);
      return {
        meta: {
          alias: null,
          disabled: false,
          negate: false
        },
        query: {
          bool: {
            should: orClauses
          }
        }
      };
    });
    const clauses = filters.map(({
      query
    }) => query);
    return {
      meta: {
        alias,
        disabled: false,
        negate: excludeExceptions
      },
      query: {
        bool: {
          should: clauses
        }
      }
    };
  }
};

exports.buildExceptionFilter = buildExceptionFilter;

const buildExclusionClause = booleanFilter => {
  return {
    bool: {
      must_not: booleanFilter
    }
  };
};

exports.buildExclusionClause = buildExclusionClause;

const buildMatchClause = entry => {
  const {
    field,
    operator,
    value
  } = entry;
  const matchClause = {
    bool: {
      minimum_should_match: 1,
      should: [{
        match_phrase: {
          [field]: value
        }
      }]
    }
  };

  if (operator === 'excluded') {
    return buildExclusionClause(matchClause);
  } else {
    return matchClause;
  }
};

exports.buildMatchClause = buildMatchClause;

const getBaseMatchAnyClause = entry => {
  const {
    field,
    value
  } = entry;

  if (value.length === 1) {
    return {
      bool: {
        minimum_should_match: 1,
        should: [{
          match_phrase: {
            [field]: value[0]
          }
        }]
      }
    };
  }

  return {
    bool: {
      minimum_should_match: 1,
      should: value.map(val => {
        return {
          bool: {
            minimum_should_match: 1,
            should: [{
              match_phrase: {
                [field]: val
              }
            }]
          }
        };
      })
    }
  };
};

exports.getBaseMatchAnyClause = getBaseMatchAnyClause;

const buildMatchAnyClause = entry => {
  const {
    operator
  } = entry;
  const matchAnyClause = getBaseMatchAnyClause(entry);

  if (operator === 'excluded') {
    return buildExclusionClause(matchAnyClause);
  } else {
    return matchAnyClause;
  }
};

exports.buildMatchAnyClause = buildMatchAnyClause;

const buildMatchWildcardClause = entry => {
  const {
    field,
    operator,
    value
  } = entry;
  const wildcardClause = {
    bool: {
      filter: {
        wildcard: {
          [field]: value
        }
      }
    }
  };

  if (operator === 'excluded') {
    return buildExclusionClause(wildcardClause);
  } else {
    return wildcardClause;
  }
};

exports.buildMatchWildcardClause = buildMatchWildcardClause;

const buildExistsClause = entry => {
  const {
    field,
    operator
  } = entry;
  const existsClause = {
    bool: {
      minimum_should_match: 1,
      should: [{
        exists: {
          field
        }
      }]
    }
  };

  if (operator === 'excluded') {
    return buildExclusionClause(existsClause);
  } else {
    return existsClause;
  }
};

exports.buildExistsClause = buildExistsClause;

const isBooleanFilter = clause => {
  const keys = Object.keys(clause);
  return keys.includes('bool') != null;
};

const getBaseNestedClause = (entries, parentField) => {
  if (entries.length === 1) {
    const [singleNestedEntry] = entries;
    const innerClause = createInnerAndClauses(singleNestedEntry, parentField);
    return isBooleanFilter(innerClause) ? innerClause : {
      bool: {}
    };
  }

  return {
    bool: {
      filter: entries.map(nestedEntry => createInnerAndClauses(nestedEntry, parentField))
    }
  };
};

exports.getBaseNestedClause = getBaseNestedClause;

const buildNestedClause = entry => {
  const {
    field,
    entries
  } = entry;
  const baseNestedClause = getBaseNestedClause(entries, field);
  return {
    nested: {
      path: field,
      query: baseNestedClause,
      score_mode: 'none'
    }
  };
};

exports.buildNestedClause = buildNestedClause;

const createInnerAndClauses = (entry, parent) => {
  const field = parent != null ? `${parent}.${entry.field}` : entry.field;

  if (_securitysolutionIoTsListTypes.entriesExists.is(entry)) {
    return buildExistsClause({ ...entry,
      field
    });
  } else if (_securitysolutionIoTsListTypes.entriesMatch.is(entry)) {
    return buildMatchClause({ ...entry,
      field
    });
  } else if (_securitysolutionIoTsListTypes.entriesMatchAny.is(entry)) {
    return buildMatchAnyClause({ ...entry,
      field
    });
  } else if (_securitysolutionIoTsListTypes.entriesMatchWildcard.is(entry)) {
    return buildMatchWildcardClause({ ...entry,
      field
    });
  } else if (_securitysolutionIoTsListTypes.entriesNested.is(entry)) {
    return buildNestedClause(entry);
  } else {
    throw new TypeError(`Unexpected exception entry: ${entry}`);
  }
};

exports.createInnerAndClauses = createInnerAndClauses;