"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isScriptedRangeFilter = exports.isRangeFilter = exports.getRangeScript = exports.getRangeFilterField = exports.buildRangeFilter = void 0;

var _lodash = require("lodash");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
const OPERANDS_IN_RANGE = 2;
const operators = {
  gt: '>',
  gte: '>=',
  lte: '<=',
  lt: '<'
};
const comparators = {
  gt: 'boolean gt(Supplier s, def v) {return s.get() > v}',
  gte: 'boolean gte(Supplier s, def v) {return s.get() >= v}',
  lte: 'boolean lte(Supplier s, def v) {return s.get() <= v}',
  lt: 'boolean lt(Supplier s, def v) {return s.get() < v}'
};
const dateComparators = {
  gt: 'boolean gt(Supplier s, def v) {return s.get().toInstant().isAfter(Instant.parse(v))}',
  gte: 'boolean gte(Supplier s, def v) {return !s.get().toInstant().isBefore(Instant.parse(v))}',
  lte: 'boolean lte(Supplier s, def v) {return !s.get().toInstant().isAfter(Instant.parse(v))}',
  lt: 'boolean lt(Supplier s, def v) {return s.get().toInstant().isBefore(Instant.parse(v))}'
};
/**
 * An interface for all possible range filter params
 * It is similar, but not identical to estypes.QueryDslRangeQuery
 * @public
 */

const hasRangeKeys = params => Boolean((0, _lodash.keys)(params).find(key => ['gte', 'gt', 'lte', 'lt', 'from', 'to'].includes(key)));

/**
 * @param filter
 * @returns `true` if a filter is an `RangeFilter`
 *
 * @public
 */
const isRangeFilter = filter => (0, _lodash.has)(filter, 'query.range');
/**
 *
 * @param filter
 * @returns `true` if a filter is a scripted `RangeFilter`
 *
 * @public
 */


exports.isRangeFilter = isRangeFilter;

const isScriptedRangeFilter = filter => {
  const params = (0, _lodash.get)(filter, 'query.script.script.params', {});
  return hasRangeKeys(params);
};
/**
 * @internal
 */


exports.isScriptedRangeFilter = isScriptedRangeFilter;

const getRangeFilterField = filter => {
  return filter.query.range && Object.keys(filter.query.range)[0];
};

exports.getRangeFilterField = getRangeFilterField;

const formatValue = params => (0, _lodash.map)(params, (val, key) => (0, _lodash.get)(operators, key) + val).join(' ');
/**
 * Creates a filter where the value for the given field is in the given range
 * params should be an object containing `lt`, `lte`, `gt`, and/or `gte`
 *
 * @param field
 * @param params
 * @param indexPattern
 * @param formattedValue
 * @returns
 *
 * @public
 */


const buildRangeFilter = (field, params, indexPattern, formattedValue) => {
  params = (0, _lodash.mapValues)(params, value => field.type === 'number' ? parseFloat(value) : value);
  if ('gte' in params && 'gt' in params) throw new Error('gte and gt are mutually exclusive');
  if ('lte' in params && 'lt' in params) throw new Error('lte and lt are mutually exclusive');
  const totalInfinite = ['gt', 'lt'].reduce((acc, op) => {
    const key = op in params ? op : `${op}e`;
    const isInfinite = Math.abs((0, _lodash.get)(params, key)) === Infinity;

    if (isInfinite) {
      acc++; // @ts-ignore

      delete params[key];
    }

    return acc;
  }, 0);
  const meta = {
    index: indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.id,
    params: {},
    field: field.name,
    ...(formattedValue ? {
      formattedValue
    } : {})
  };

  if (totalInfinite === OPERANDS_IN_RANGE) {
    return {
      meta,
      query: {
        match_all: {}
      }
    };
  } else if (field.scripted) {
    const scr = getRangeScript(field, params); // TODO: type mismatch enforced

    scr.script.params.value = formatValue(scr.script.params);
    return {
      meta,
      query: {
        script: scr
      }
    };
  } else {
    return {
      meta,
      query: {
        range: {
          [field.name]: params
        }
      }
    };
  }
};
/**
 * @internal
 */


exports.buildRangeFilter = buildRangeFilter;

const getRangeScript = (field, params) => {
  const knownParams = (0, _lodash.mapValues)((0, _lodash.pickBy)(params, (val, key) => key in operators), value => field.type === 'number' && typeof value === 'string' ? parseFloat(value) : value);
  let script = (0, _lodash.map)(knownParams, (_, key) => '(' + field.script + ')' + (0, _lodash.get)(operators, key) + key).join(' && '); // We must wrap painless scripts in a lambda in case they're more than a simple expression

  if (field.lang === 'painless') {
    const comp = field.type === 'date' ? dateComparators : comparators;
    const currentComparators = (0, _lodash.reduce)(knownParams, (acc, val, key) => acc.concat((0, _lodash.get)(comp, key)), []).join(' ');
    const comparisons = (0, _lodash.map)(knownParams, (val, key) => `${key}(() -> { ${field.script} }, params.${key})`).join(' && ');
    script = `${currentComparators}${comparisons}`;
  }

  return {
    script: {
      source: script,
      params: knownParams,
      lang: field.lang
    }
  };
};

exports.getRangeScript = getRangeScript;