"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addSyntheticsMonitorRoute = void 0;

var _configSchema = require("@kbn/config-schema");

var _runtime_types = require("../../../common/runtime_types");

var _constants = require("../../../common/constants");

var _synthetics_monitor = require("../../lib/saved_objects/synthetics_monitor");

var _monitor_validation = require("./monitor_validation");

var _monitor_upgrade_sender = require("./telemetry/monitor_upgrade_sender");

var _secrets = require("../../lib/synthetics_service/utils/secrets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */


const addSyntheticsMonitorRoute = () => ({
  method: 'POST',
  path: _constants.API_URLS.SYNTHETICS_MONITORS,
  validate: {
    body: _configSchema.schema.any()
  },
  handler: async ({
    request,
    response,
    savedObjectsClient,
    server
  }) => {
    const monitor = request.body;
    const validationResult = (0, _monitor_validation.validateMonitor)(monitor);

    if (!validationResult.valid) {
      const {
        reason: message,
        details,
        payload
      } = validationResult;
      return response.badRequest({
        body: {
          message,
          attributes: {
            details,
            ...payload
          }
        }
      });
    }

    const newMonitor = await savedObjectsClient.create(_synthetics_monitor.syntheticsMonitorType, (0, _secrets.formatSecrets)({ ...monitor,
      revision: 1
    }));
    const {
      syntheticsService
    } = server;
    const errors = await syntheticsService.addConfig({ ...monitor,
      id: newMonitor.id,
      fields: {
        config_id: newMonitor.id
      },
      fields_under_root: true
    });
    (0, _monitor_upgrade_sender.sendTelemetryEvents)(server.logger, server.telemetry, (0, _monitor_upgrade_sender.formatTelemetryEvent)({
      monitor: newMonitor,
      errors,
      isInlineScript: Boolean(monitor[_runtime_types.ConfigKey.SOURCE_INLINE]),
      kibanaVersion: server.kibanaVersion
    }));

    if (errors && errors.length > 0) {
      return response.ok({
        body: {
          message: 'error pushing monitor to the service',
          attributes: {
            errors
          }
        }
      });
    }

    return response.ok({
      body: newMonitor
    });
  }
});

exports.addSyntheticsMonitorRoute = addSyntheticsMonitorRoute;