"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.editRule = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _lodash = require("lodash");

var _securitysolutionIoTsUtils = require("@kbn/securitysolution-io-ts-utils");

var _rule_schemas = require("../schemas/rule_schemas");

var _add_tags = require("./add_tags");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */


class EditRuleError extends Error {
  constructor(message, statusCode) {
    super(message);
    (0, _defineProperty2.default)(this, "statusCode", void 0);
    this.statusCode = statusCode;
  }

}
/** At this point we support editing of only these fields. */


const FIELDS_THAT_CAN_BE_EDITED = ['params', 'tags'];
/**
 * Applies in-memory modifications to a given rule and updates it in Elasticsearch via RulesClient.
 *
 * NOTE: At this point we only support editing of the following fields:
 *   - rule.params
 *   - rule.tags
 * All other changes made by the `edit` function will be ignored.
 *
 * @returns The edited rule.
 */

const editRule = async params => {
  const {
    rulesClient,
    rule,
    edit
  } = params;
  const isPrebuiltRule = rule.params.immutable;
  const isCustomRule = !rule.params.immutable;

  if (isPrebuiltRule) {
    throw new EditRuleError('Elastic rule can`t be edited', 400);
  }

  const editedRule = applyChanges(rule, edit); // If the rule wasn't changed by the `edit` function, we don't need to proceed with the update.

  if (!isRuleChanged(rule, editedRule)) {
    return rule;
  } // We need to increment the rule's version if it is a custom rule. If the rule is an Elastic
  // prebuilt rule, we don't want to touch its version - it's managed by the rule authors.
  // This check is left here explicitly because we're planning to allow editing for prebuilt rules,
  // and the check for isPrebuiltRule above might be removed.


  if (isCustomRule) {
    editedRule.params.version = editedRule.params.version + 1;
  }

  const updateData = createUpdateData(rule, editedRule);
  await rulesClient.update({
    id: rule.id,
    data: updateData
  }); // It would be great to return the updated rule returned from the RulesClient.update() call.
  // Note that there's a type mismatch between RuleAlertType and the update method result.

  return editedRule;
};

exports.editRule = editRule;

const applyChanges = (originalRule, edit) => {
  // For safety, deeply clone the rule object before applying edits to it.
  const clonedRule = (0, _lodash.cloneDeep)(originalRule);
  const editedRule = edit(clonedRule);
  const sanitizedRule = validateAndSanitizeChanges(originalRule, editedRule);
  return sanitizedRule;
};

const validateAndSanitizeChanges = (original, changed) => {
  // These checks should never throw unless there's a bug in the passed `edit` function.
  if (changed.params.immutable !== original.params.immutable) {
    throw new EditRuleError(`Internal rule editing error: can't change "params.immutable"`, 500);
  }

  if (changed.params.version !== original.params.version) {
    throw new EditRuleError(`Internal rule editing error: can't change "params.version"`, 500);
  }

  return { ...changed,
    tags: (0, _add_tags.addTags)(changed.tags, changed.params.ruleId, changed.params.immutable)
  };
};

const isRuleChanged = (originalRule, editedRule) => {
  const originalData = (0, _lodash.pick)(originalRule, FIELDS_THAT_CAN_BE_EDITED);
  const editedData = (0, _lodash.pick)(editedRule, FIELDS_THAT_CAN_BE_EDITED);
  return !(0, _lodash.isEqual)(originalData, editedData);
};

const createUpdateData = (originalRule, editedRule) => {
  const data = { // At this point we intentionally support updating of only these fields:
    ...(0, _lodash.pick)(editedRule, FIELDS_THAT_CAN_BE_EDITED),
    // We omit other fields and get them from the original (unedited) rule:
    name: originalRule.name,
    schedule: originalRule.schedule,
    actions: originalRule.actions,
    throttle: originalRule.throttle,
    notifyWhen: originalRule.notifyWhen
  };
  const [validatedData, validationError] = (0, _securitysolutionIoTsUtils.validate)(data, _rule_schemas.internalRuleUpdate);

  if (validationError != null || validatedData === null) {
    throw new EditRuleError(`Editing rule would create invalid rule: ${validationError}`, 500);
  }

  return validatedData;
};