"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ScreenshotObservableHandler = void 0;

var _rxjs = require("rxjs");

var _operators = require("rxjs/operators");

var _common = require("../../common");

var _browsers = require("../browsers");

var _get_element_position_data = require("./get_element_position_data");

var _get_number_of_items = require("./get_number_of_items");

var _get_render_errors = require("./get_render_errors");

var _get_screenshots = require("./get_screenshots");

var _get_time_range = require("./get_time_range");

var _inject_css = require("./inject_css");

var _open_url = require("./open_url");

var _wait_for_render = require("./wait_for_render");

var _wait_for_visualizations = require("./wait_for_visualizations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */


const getDefaultElementPosition = dimensions => {
  const height = (dimensions === null || dimensions === void 0 ? void 0 : dimensions.height) || _browsers.DEFAULT_VIEWPORT.height;
  const width = (dimensions === null || dimensions === void 0 ? void 0 : dimensions.width) || _browsers.DEFAULT_VIEWPORT.width;
  return [{
    position: {
      boundingClientRect: {
        top: 0,
        left: 0,
        height,
        width
      },
      scroll: {
        x: 0,
        y: 0
      }
    },
    attributes: {}
  }];
};

class ScreenshotObservableHandler {
  constructor(driver, logger, layout, options) {
    this.driver = driver;
    this.logger = logger;
    this.layout = layout;
    this.options = options;
  }
  /*
   * Decorates a TimeoutError with context of the phase that has timed out.
   */


  waitUntil(timeoutValue, label) {
    return source => source.pipe((0, _operators.catchError)(error => {
      throw new Error(`The "${label}" phase encountered an error: ${error}`);
    }), (0, _operators.timeoutWith)(timeoutValue, (0, _rxjs.throwError)(new Error(`The "${label}" phase took longer than ${timeoutValue / 1000} seconds.`))));
  }

  openUrl(index, urlOrUrlWithContext) {
    return (0, _rxjs.defer)(() => {
      var _context, _this$options$headers;

      let url;
      let context;

      if (typeof urlOrUrlWithContext === 'string') {
        url = urlOrUrlWithContext;
      } else {
        [url, context] = urlOrUrlWithContext;
      }

      return (0, _open_url.openUrl)(this.driver, this.logger, this.options.timeouts.openUrl, index, url, { ...((_context = context) !== null && _context !== void 0 ? _context : {}),
        layout: this.layout.id
      }, (_this$options$headers = this.options.headers) !== null && _this$options$headers !== void 0 ? _this$options$headers : {});
    }).pipe(this.waitUntil(this.options.timeouts.openUrl, 'open URL'));
  }

  waitForElements() {
    const driver = this.driver;
    const waitTimeout = this.options.timeouts.waitForElements;
    return (0, _rxjs.defer)(() => (0, _get_number_of_items.getNumberOfItems)(driver, this.logger, waitTimeout, this.layout)).pipe((0, _operators.mergeMap)(itemsCount => (0, _wait_for_visualizations.waitForVisualizations)(driver, this.logger, waitTimeout, itemsCount, this.layout)), this.waitUntil(waitTimeout, 'wait for elements'));
  }

  completeRender(apmTrans) {
    const driver = this.driver;
    const layout = this.layout;
    const logger = this.logger;
    return (0, _rxjs.defer)(async () => {
      var _layout$positionEleme; // Waiting till _after_ elements have rendered before injecting our CSS
      // allows for them to be displayed properly in many cases


      await (0, _inject_css.injectCustomCss)(driver, logger, layout);
      const apmPositionElements = apmTrans === null || apmTrans === void 0 ? void 0 : apmTrans.startSpan('position-elements', 'correction'); // position panel elements for print layout

      await ((_layout$positionEleme = layout.positionElements) === null || _layout$positionEleme === void 0 ? void 0 : _layout$positionEleme.call(layout, driver, logger));
      apmPositionElements === null || apmPositionElements === void 0 ? void 0 : apmPositionElements.end();
      await (0, _wait_for_render.waitForRenderComplete)(driver, logger, this.options.timeouts.loadDelay, layout);
    }).pipe((0, _operators.mergeMap)(() => (0, _rxjs.forkJoin)({
      timeRange: (0, _get_time_range.getTimeRange)(driver, logger, layout),
      elementsPositionAndAttributes: (0, _get_element_position_data.getElementPositionAndAttributes)(driver, logger, layout),
      renderErrors: (0, _get_render_errors.getRenderErrors)(driver, logger, layout)
    })), this.waitUntil(this.options.timeouts.renderComplete, 'render complete'));
  }

  setupPage(index, url, apmTrans) {
    return this.openUrl(index, url).pipe((0, _operators.switchMapTo)(this.waitForElements()), (0, _operators.switchMapTo)(this.completeRender(apmTrans)));
  }

  getScreenshots() {
    return withRenderComplete => withRenderComplete.pipe((0, _operators.mergeMap)(async data => {
      var _data$elementsPositio;

      this.checkPageIsOpen(); // fail the report job if the browser has closed

      const elements = (_data$elementsPositio = data.elementsPositionAndAttributes) !== null && _data$elementsPositio !== void 0 ? _data$elementsPositio : getDefaultElementPosition(this.layout.getViewport());
      let screenshots = [];

      try {
        screenshots = await (0, _get_screenshots.getScreenshots)(this.driver, this.logger, elements, this.layout);
      } catch (e) {
        throw new _common.errors.FailedToCaptureScreenshot(e.message);
      }

      const {
        timeRange,
        error: setupError,
        renderErrors
      } = data;
      return {
        timeRange,
        screenshots,
        error: setupError,
        renderErrors,
        elementsPositionAndAttributes: elements
      };
    }));
  }

  checkPageIsOpen() {
    if (!this.driver.isPageOpen()) {
      throw (0, _browsers.getChromiumDisconnectedError)();
    }
  }

}

exports.ScreenshotObservableHandler = ScreenshotObservableHandler;