"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toPdf = toPdf;

var _lodash = require("lodash");

var _common = require("../../../common");

var _pdf_maker = require("./pdf_maker");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */


const supportedLayouts = [_common.LayoutTypes.PRESERVE_LAYOUT, _common.LayoutTypes.CANVAS, _common.LayoutTypes.PRINT];
/**
 * PDFs can be a single, long page or they can be multiple pages. For example:
 *
 * => When creating a PDF intended for print multiple PNGs will be spread out across pages
 * => When creating a PDF from a Canvas workpad, each page in the workpad will be placed on a separate page
 */

const getTimeRange = urlScreenshots => {
  const grouped = (0, _lodash.groupBy)(urlScreenshots.map(u => u.timeRange));
  const values = Object.values(grouped);

  if (values.length === 1) {
    return values[0][0];
  }

  return null;
};

function toPdf({
  title,
  logo,
  logger,
  packageInfo
}) {
  return async screenshotResult => {
    const timeRange = getTimeRange(screenshotResult.results);

    try {
      const {
        buffer,
        pageCount
      } = await (0, _pdf_maker.pngsToPdf)({
        results: screenshotResult.results,
        title: title ? title + (timeRange ? ` - ${timeRange}` : '') : undefined,
        packageInfo,
        logo,
        layout: screenshotResult.layout,
        logger
      });
      return { ...screenshotResult,
        metrics: {
          pageCount,
          ...screenshotResult.metrics
        },
        result: {
          data: buffer,
          errors: screenshotResult.results.flatMap(result => result.error ? [result.error] : []),
          renderErrors: screenshotResult.results.flatMap(result => result.renderErrors ? [...result.renderErrors] : [])
        }
      };
    } catch (e) {
      logger.error(`Could not generate the PDF buffer!`);
      throw e;
    }
  };
}