"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findingsInputSchema = exports.defineFindingsIndexRoute = exports.DEFAULT_FINDINGS_PER_PAGE = void 0;

var _esQuery = require("@kbn/es-query");

var _configSchema = require("@kbn/config-schema");

var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");

var _get_latest_cycle_ids = require("./get_latest_cycle_ids");

var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */


const DEFAULT_FINDINGS_PER_PAGE = 20;
exports.DEFAULT_FINDINGS_PER_PAGE = DEFAULT_FINDINGS_PER_PAGE;

const getPointerForFirstDoc = (page, perPage) => page <= 1 ? 0 : page * perPage - perPage;

const getSort = (sortField, sortOrder) => sortField ? {
  sort: [{
    [sortField]: sortOrder
  }]
} : {
  sort: [{
    '@timestamp': {
      order: sortOrder
    }
  }]
};

const getSearchFields = fields => fields ? {
  _source: fields.split(',')
} : {};

const getFindingsEsQuery = (query, options) => {
  return {
    index: _constants.CSP_KUBEBEAT_INDEX_PATTERN,
    query,
    ...options
  };
};

const buildLatestCycleFilter = (filter, latestCycleIds) => {
  if (!!latestCycleIds) {
    filter.push({
      terms: {
        'cycle_id.keyword': latestCycleIds
      }
    });
  }

  return filter;
};

const convertKqueryToElasticsearchQuery = (kquery, logger) => {
  let dslFilterQuery;

  try {
    dslFilterQuery = kquery ? (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(kquery)) : [];

    if (!Array.isArray(dslFilterQuery)) {
      dslFilterQuery = [dslFilterQuery];
    }
  } catch (err) {
    logger.warn(`Invalid kuery syntax for the filter (${kquery}) error: ${err.message}`);
    throw err;
  }

  return dslFilterQuery;
};

const buildQueryRequest = (kquery, latestCycleIds, logger) => {
  const kqueryFilter = convertKqueryToElasticsearchQuery(kquery, logger);
  const filter = buildLatestCycleFilter(kqueryFilter, latestCycleIds);
  const query = {
    bool: {
      filter
    }
  };
  return query;
};

const buildOptionsRequest = queryParams => ({
  size: queryParams.per_page,
  from: getPointerForFirstDoc(queryParams.page, queryParams.per_page),
  ...getSort(queryParams.sort_field, queryParams.sort_order),
  ...getSearchFields(queryParams.fields)
});

const defineFindingsIndexRoute = (router, cspContext) => router.get({
  path: _constants.FINDINGS_ROUTE_PATH,
  validate: {
    query: findingsInputSchema
  }
}, async (context, request, response) => {
  if (!context.fleet.authz.fleet.all) {
    return response.forbidden();
  }

  try {
    const esClient = context.core.elasticsearch.client.asCurrentUser;
    const options = buildOptionsRequest(request.query);
    const latestCycleIds = request.query.latest_cycle === true ? await (0, _get_latest_cycle_ids.getLatestCycleIds)(esClient, cspContext.logger) : undefined;

    if (request.query.latest_cycle === true && latestCycleIds === undefined) {
      return response.ok({
        body: []
      });
    }

    const query = buildQueryRequest(request.query.kquery, latestCycleIds, cspContext.logger);
    const esQuery = getFindingsEsQuery(query, options);
    const findings = await esClient.search(esQuery);
    const hits = findings.hits.hits;
    return response.ok({
      body: hits
    });
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    cspContext.logger.error(`Failed to fetch Findings ${error.message}`);
    return response.customError({
      body: {
        message: error.message
      },
      statusCode: error.statusCode
    });
  }
});

exports.defineFindingsIndexRoute = defineFindingsIndexRoute;

const findingsInputSchema = _configSchema.schema.object({
  /**
   * The page of objects to return
   */
  page: _configSchema.schema.number({
    defaultValue: 1,
    min: 1
  }),

  /**
   * The number of objects to include in each page
   */
  per_page: _configSchema.schema.number({
    defaultValue: DEFAULT_FINDINGS_PER_PAGE,
    min: 0
  }),

  /**
   * Boolean flag to indicate for receiving only the latest findings
   */
  latest_cycle: _configSchema.schema.maybe(_configSchema.schema.boolean()),

  /**
   * The field to use for sorting the found objects.
   */
  sort_field: _configSchema.schema.maybe(_configSchema.schema.string()),

  /**
   * The order to sort by
   */
  sort_order: _configSchema.schema.oneOf([_configSchema.schema.literal('asc'), _configSchema.schema.literal('desc')], {
    defaultValue: 'desc'
  }),

  /**
   * The fields in the entity to return in the response
   */
  fields: _configSchema.schema.maybe(_configSchema.schema.string()),

  /**
   * kql query
   */
  kquery: _configSchema.schema.maybe(_configSchema.schema.string())
});

exports.findingsInputSchema = findingsInputSchema;