"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getHistogramRangeSteps = exports.getHistogramIntervalRequest = exports.fetchTransactionDurationHistogramRangeSteps = void 0;

var _d3Scale = require("d3-scale");

var _elasticsearch_fieldnames = require("../../../../common/elasticsearch_fieldnames");

var _get_query_with_params = require("./get_query_with_params");

var _get_request_base = require("./get_request_base");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */


const getHistogramRangeSteps = (min, max, steps) => {
  // A d3 based scale function as a helper to get equally distributed bins on a log scale.
  // We round the final values because the ES range agg we use won't accept numbers with decimals for `transaction.duration.us`.
  const logFn = (0, _d3Scale.scaleLog)().domain([min, max]).range([1, steps]);
  return [...Array(steps).keys()].map(logFn.invert).map(d => isNaN(d) ? 0 : Math.round(d));
};

exports.getHistogramRangeSteps = getHistogramRangeSteps;

const getHistogramIntervalRequest = params => ({ ...(0, _get_request_base.getRequestBase)(params),
  body: {
    query: (0, _get_query_with_params.getQueryWithParams)({
      params
    }),
    size: 0,
    aggs: {
      transaction_duration_min: {
        min: {
          field: _elasticsearch_fieldnames.TRANSACTION_DURATION
        }
      },
      transaction_duration_max: {
        max: {
          field: _elasticsearch_fieldnames.TRANSACTION_DURATION
        }
      }
    }
  }
});

exports.getHistogramIntervalRequest = getHistogramIntervalRequest;

const fetchTransactionDurationHistogramRangeSteps = async (esClient, params) => {
  const steps = 100;
  const resp = await esClient.search(getHistogramIntervalRequest(params));

  if (resp.hits.total.value === 0) {
    return getHistogramRangeSteps(0, 1, 100);
  }

  if (resp.aggregations === undefined) {
    throw new Error('fetchTransactionDurationHistogramRangeSteps failed, did not return aggregations.');
  }

  const min = resp.aggregations.transaction_duration_min.value;
  const max = resp.aggregations.transaction_duration_max.value * 2;
  return getHistogramRangeSteps(min, max, steps);
};

exports.fetchTransactionDurationHistogramRangeSteps = fetchTransactionDurationHistogramRangeSteps;