"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setupOptionsListSuggestionsRoute = void 0;

var _lodash = require("lodash");

var _configSchema = require("@kbn/config-schema");

var _server = require("../../../../kibana_utils/server");

var _common = require("../../../../data_views/common");

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
const setupOptionsListSuggestionsRoute = ({
  http
}, getAutocompleteSettings) => {
  const router = http.createRouter();
  router.post({
    path: '/api/kibana/controls/optionsList/{index}',
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      }, {
        unknowns: 'allow'
      }),
      body: _configSchema.schema.object({
        fieldName: _configSchema.schema.string(),
        filters: _configSchema.schema.maybe(_configSchema.schema.any()),
        fieldSpec: _configSchema.schema.maybe(_configSchema.schema.any()),
        searchString: _configSchema.schema.maybe(_configSchema.schema.string()),
        selectedOptions: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
      }, {
        unknowns: 'allow'
      })
    }
  }, async (context, request, response) => {
    try {
      const suggestionRequest = request.body;
      const {
        index
      } = request.params;
      const esClient = context.core.elasticsearch.client.asCurrentUser;
      const suggestionsResponse = await getOptionsListSuggestions({
        abortedEvent$: request.events.aborted$,
        request: suggestionRequest,
        esClient,
        index
      });
      return response.ok({
        body: suggestionsResponse
      });
    } catch (e) {
      const kbnErr = (0, _server.getKbnServerError)(e);
      return (0, _server.reportServerError)(response, kbnErr);
    }
  });

  const getOptionsListSuggestions = async ({
    abortedEvent$,
    esClient,
    request,
    index
  }) => {
    var _get, _Object$entries, _Object$entries$filte;

    const abortController = new AbortController();
    abortedEvent$.subscribe(() => abortController.abort());
    const {
      fieldName,
      searchString,
      selectedOptions,
      filters,
      fieldSpec
    } = request;
    const body = getOptionsListBody(fieldName, fieldSpec, searchString, selectedOptions, filters);
    const rawEsResult = await esClient.search({
      index,
      body
    }, {
      signal: abortController.signal
    }); // parse raw ES response into OptionsListSuggestionResponse

    const totalCardinality = (0, _lodash.get)(rawEsResult, 'aggregations.unique_terms.value');
    const suggestions = (_get = (0, _lodash.get)(rawEsResult, 'aggregations.suggestions.buckets')) === null || _get === void 0 ? void 0 : _get.map(suggestion => (fieldSpec === null || fieldSpec === void 0 ? void 0 : fieldSpec.type) === 'string' ? suggestion.key : suggestion.key_as_string);
    const rawInvalidSuggestions = (0, _lodash.get)(rawEsResult, 'aggregations.validation.buckets');
    const invalidSelections = rawInvalidSuggestions && !(0, _lodash.isEmpty)(rawInvalidSuggestions) ? (_Object$entries = Object.entries(rawInvalidSuggestions)) === null || _Object$entries === void 0 ? void 0 : (_Object$entries$filte = _Object$entries.filter(([, value]) => (value === null || value === void 0 ? void 0 : value.doc_count) === 0)) === null || _Object$entries$filte === void 0 ? void 0 : _Object$entries$filte.map(([key]) => key) : undefined;
    return {
      suggestions,
      totalCardinality,
      invalidSelections
    };
  };

  const getOptionsListBody = (fieldName, fieldSpec, searchString, selectedOptions, filters = []) => {
    // https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-regexp-query.html#_standard_operators
    const getEscapedQuery = (q = '') => q.replace(/[.?+*|{}[\]()"\\#@&<>~]/g, match => `\\${match}`); // Helps ensure that the regex is not evaluated eagerly against the terms dictionary


    const executionHint = 'map'; // Suggestions

    const shardSize = 10;
    const suggestionsAgg = {
      terms: {
        field: fieldName,
        // terms on boolean fields don't support include
        ...((fieldSpec === null || fieldSpec === void 0 ? void 0 : fieldSpec.type) !== 'boolean' && {
          include: `${getEscapedQuery(searchString !== null && searchString !== void 0 ? searchString : '')}.*`
        }),
        execution_hint: executionHint,
        shard_size: shardSize
      }
    }; // Validation

    const selectedOptionsFilters = selectedOptions === null || selectedOptions === void 0 ? void 0 : selectedOptions.reduce((acc, currentOption) => {
      acc[currentOption] = {
        match: {
          [fieldName]: currentOption
        }
      };
      return acc;
    }, {});
    const validationAgg = selectedOptionsFilters && !(0, _lodash.isEmpty)(selectedOptionsFilters) ? {
      filters: {
        filters: selectedOptionsFilters
      }
    } : undefined;
    const {
      terminateAfter,
      timeout
    } = getAutocompleteSettings();
    const body = {
      size: 0,
      timeout: `${timeout}ms`,
      terminate_after: terminateAfter,
      query: {
        bool: {
          filter: filters
        }
      },
      aggs: {
        suggestions: suggestionsAgg,
        ...(validationAgg ? {
          validation: validationAgg
        } : {}),
        unique_terms: {
          cardinality: {
            field: fieldName
          }
        }
      }
    };
    const subTypeNested = fieldSpec && (0, _common.getFieldSubtypeNested)(fieldSpec);

    if (subTypeNested) {
      return { ...body,
        aggs: {
          nestedSuggestions: {
            nested: {
              path: subTypeNested.nested.path
            },
            aggs: body.aggs
          }
        }
      };
    }

    return body;
  };
};

exports.setupOptionsListSuggestionsRoute = setupOptionsListSuggestionsRoute;