import _objectSpread from "@babel/runtime/helpers/objectSpread2";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { nodeTypes } from '../node_types';
import * as ast from '../ast';
import { getRangeScript } from '../../filters';
import { getFields } from './utils/get_fields';
import { getDataViewFieldSubtypeNested, getTimeZoneFromSettings } from '../../utils';
import { getFullFieldNameNode } from './utils/get_full_field_name_node';
export function buildNodeParams(fieldName, operator, value) {
  // Run through the parser instead treating it as a literal because it may contain wildcards
  var fieldNameArg = ast.fromLiteralExpression(fieldName);
  var valueArg = nodeTypes.literal.buildNode(value);
  return {
    arguments: [fieldNameArg, operator, valueArg]
  };
}
export function toElasticsearchQuery(node, indexPattern) {
  var config = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  var context = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};

  var _node$arguments = _slicedToArray(node.arguments, 3),
      fieldNameArg = _node$arguments[0],
      operatorArg = _node$arguments[1],
      valueArg = _node$arguments[2];

  var fullFieldNameArg = getFullFieldNameNode(fieldNameArg, indexPattern, context !== null && context !== void 0 && context.nested ? context.nested.path : undefined);
  var fields = indexPattern ? getFields(fullFieldNameArg, indexPattern) : []; // If no fields are found in the index pattern we send through the given field name as-is. We do this to preserve
  // the behaviour of lucene on dashboards where there are panels based on different index patterns that have different
  // fields. If a user queries on a field that exists in one pattern but not the other, the index pattern without the
  // field should return no results. It's debatable whether this is desirable, but it's been that way forever, so we'll
  // keep things familiar for now.

  if (fields && fields.length === 0) {
    fields.push({
      name: ast.toElasticsearchQuery(fullFieldNameArg),
      scripted: false,
      type: ''
    });
  }

  var queries = fields.map(function (field) {
    var wrapWithNestedQuery = function wrapWithNestedQuery(query) {
      // Wildcards can easily include nested and non-nested fields. There isn't a good way to let
      // users handle this themselves so we automatically add nested queries in this scenario.
      var subTypeNested = getDataViewFieldSubtypeNested(field);

      if (!(fullFieldNameArg.type === 'wildcard') || !(subTypeNested !== null && subTypeNested !== void 0 && subTypeNested.nested) || context.nested) {
        return query;
      } else {
        return {
          nested: {
            path: subTypeNested.nested.path,
            query: query,
            score_mode: 'none'
          }
        };
      }
    };

    var queryParams = _defineProperty({}, operatorArg, ast.toElasticsearchQuery(valueArg));

    if (field.scripted) {
      return {
        script: getRangeScript(field, queryParams)
      };
    } else if (field.type === 'date') {
      var timeZoneParam = config.dateFormatTZ ? {
        time_zone: getTimeZoneFromSettings(config.dateFormatTZ)
      } : {};
      return wrapWithNestedQuery({
        range: _defineProperty({}, field.name, _objectSpread(_objectSpread({}, queryParams), timeZoneParam))
      });
    }

    return wrapWithNestedQuery({
      range: _defineProperty({}, field.name, queryParams)
    });
  });
  return {
    bool: {
      should: queries,
      minimum_should_match: 1
    }
  };
}