"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getClustersQuery = exports.getClustersFromAggs = exports.getClusters = void 0;

var _get_resources_types = require("./get_resources_types");

var _constants = require("../../../common/constants");

var _get_stats = require("./get_stats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */


const getClustersQuery = query => ({
  index: _constants.CSP_KUBEBEAT_INDEX_PATTERN,
  size: 0,
  query,
  aggs: {
    aggs_by_cluster_id: {
      terms: {
        field: 'cluster_id.keyword'
      },
      aggs: {
        benchmarks: {
          terms: {
            field: 'rule.benchmark.name.keyword'
          }
        },
        timestamps: {
          terms: {
            field: '@timestamp',
            size: 1,
            order: {
              _key: 'desc'
            }
          }
        },
        ..._get_resources_types.resourceTypeAggQuery,
        ..._get_stats.findingsEvaluationAggsQuery
      }
    }
  }
});

exports.getClustersQuery = getClustersQuery;

const getClustersFromAggs = clusters => clusters.map(cluster => {
  // get cluster's meta data
  const benchmarks = cluster.benchmarks.buckets;
  if (!Array.isArray(benchmarks)) throw new Error('missing aggs by benchmarks per cluster');
  const timestamps = cluster.timestamps.buckets;
  if (!Array.isArray(timestamps)) throw new Error('missing aggs by timestamps per cluster');
  const meta = {
    clusterId: cluster.key,
    benchmarkName: benchmarks[0].key,
    lastUpdate: timestamps[0].key
  }; // get cluster's stats

  if (!cluster.failed_findings || !cluster.passed_findings) throw new Error('missing findings evaluations per cluster');
  const stats = (0, _get_stats.getStatsFromFindingsEvaluationsAggs)(cluster); // get cluster's resource types aggs

  const resourcesTypesAggs = cluster.aggs_by_resource_type.buckets;
  if (!Array.isArray(resourcesTypesAggs)) throw new Error('missing aggs by resource type per cluster');
  const resourcesTypes = (0, _get_resources_types.getResourceTypeFromAggs)(resourcesTypesAggs);
  return {
    meta,
    stats,
    resourcesTypes
  };
});

exports.getClustersFromAggs = getClustersFromAggs;

const getClusters = async (esClient, query) => {
  var _queryResult$body$agg;

  const queryResult = await esClient.search(getClustersQuery(query), {
    meta: true
  });
  const clusters = (_queryResult$body$agg = queryResult.body.aggregations) === null || _queryResult$body$agg === void 0 ? void 0 : _queryResult$body$agg.aggs_by_cluster_id.buckets;
  if (!Array.isArray(clusters)) throw new Error('missing aggs by cluster id');
  return getClustersFromAggs(clusters);
};

exports.getClusters = getClusters;