"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getLatestFindingQuery = exports.defineGetComplianceDashboardRoute = void 0;

var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");

var _constants = require("../../../common/constants");

var _get_resources_types = require("./get_resources_types");

var _get_clusters = require("./get_clusters");

var _get_stats = require("./get_stats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */


const getLatestFindingQuery = () => ({
  index: _constants.CSP_KUBEBEAT_INDEX_PATTERN,
  size: 0,
  query: {
    match_all: {}
  },
  aggs: {
    aggs_by_cluster_id: {
      terms: {
        field: 'cluster_id.keyword'
      },
      aggs: {
        ordered_top_hits: {
          top_hits: {
            size: 1,
            sort: {
              '@timestamp': {
                order: 'desc'
              }
            }
          }
        }
      }
    }
  }
});

exports.getLatestFindingQuery = getLatestFindingQuery;

const getLatestCyclesIds = async esClient => {
  var _queryResult$body$agg;

  const queryResult = await esClient.search(getLatestFindingQuery(), {
    meta: true
  });
  const clusters = (_queryResult$body$agg = queryResult.body.aggregations) === null || _queryResult$body$agg === void 0 ? void 0 : _queryResult$body$agg.aggs_by_cluster_id.buckets;
  if (!Array.isArray(clusters)) throw new Error('missing aggs by cluster id');
  return clusters.map(c => {
    const topHit = c.ordered_top_hits.hits.hits[0];
    if (!topHit) throw new Error('missing cluster latest hit');
    return topHit._source.cycle_id;
  });
}; // TODO: Utilize ES "Point in Time" feature https://www.elastic.co/guide/en/elasticsearch/reference/current/point-in-time-api.html


const defineGetComplianceDashboardRoute = (router, cspContext) => router.get({
  path: _constants.STATS_ROUTE_PATH,
  validate: false
}, async (context, _, response) => {
  try {
    const esClient = context.core.elasticsearch.client.asCurrentUser;
    const latestCyclesIds = await getLatestCyclesIds(esClient);
    const query = {
      bool: {
        should: latestCyclesIds.map(id => ({
          match: {
            'cycle_id.keyword': {
              query: id
            }
          }
        }))
      }
    };
    const [stats, resourcesTypes, clusters] = await Promise.all([(0, _get_stats.getStats)(esClient, query), (0, _get_resources_types.getResourcesTypes)(esClient, query), (0, _get_clusters.getClusters)(esClient, query)]);
    const body = {
      stats,
      resourcesTypes,
      clusters
    };
    return response.ok({
      body
    });
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    return response.customError({
      body: {
        message: error.message
      },
      statusCode: error.statusCode
    });
  }
});

exports.defineGetComplianceDashboardRoute = defineGetComplianceDashboardRoute;