"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseKibanaPlatformPlugin = parseKibanaPlatformPlugin;

var _path = _interopRequireDefault(require("path"));

var _loadJsonFile = _interopRequireDefault(require("load-json-file"));

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
function isValidDepsDeclaration(input, type) {
  if (typeof input === 'undefined') return [];

  if (Array.isArray(input) && input.every(i => typeof i === 'string')) {
    return input;
  }

  throw new TypeError(`The "${type}" in plugin manifest should be an array of strings.`);
}

function parseKibanaPlatformPlugin(manifestPath) {
  if (!_path.default.isAbsolute(manifestPath)) {
    throw new TypeError('expected new platform manifest path to be absolute');
  }

  const manifest = _loadJsonFile.default.sync(manifestPath);

  if (!manifest || typeof manifest !== 'object' || Array.isArray(manifest)) {
    throw new TypeError('expected new platform plugin manifest to be a JSON encoded object');
  }

  if (typeof manifest.id !== 'string') {
    throw new TypeError('expected new platform plugin manifest to have a string id');
  }

  if (typeof manifest.version !== 'string') {
    throw new TypeError('expected new platform plugin manifest to have a string version');
  }

  if (!manifest.owner || typeof manifest.owner.name !== 'string') {
    throw new TypeError(`Expected plugin ${manifest.id} manifest to have an owner with name specified (${manifestPath})`);
  }

  return {
    directory: _path.default.dirname(manifestPath),
    manifestPath,
    manifest: { ...manifest,
      ui: !!manifest.ui,
      server: !!manifest.server,
      id: manifest.id,
      version: manifest.version,
      kibanaVersion: manifest.kibanaVersion || manifest.version,
      serviceFolders: manifest.serviceFolders || [],
      owner: manifest.owner,
      description: manifest.description,
      enabledOnAnonymousPages: Boolean(manifest.enabledOnAnonymousPages),
      requiredPlugins: isValidDepsDeclaration(manifest.requiredPlugins, 'requiredPlugins'),
      optionalPlugins: isValidDepsDeclaration(manifest.optionalPlugins, 'optionalPlugins'),
      requiredBundles: isValidDepsDeclaration(manifest.requiredBundles, 'requiredBundles'),
      extraPublicDirs: isValidDepsDeclaration(manifest.extraPublicDirs, 'extraPublicDirs')
    }
  };
}